#! /bin/csh -f
#
#	Run spotfinder in the standard way
#
#
set scoredir = /share/scores


# interpret command line
set imgfile = "$1"
foreach arg ( $* )
    if("$arg" =~ *.img) then
	set imgfile = "$arg"
    endif
    if("$arg" =~ *.spt) then
	set user_spt = "$arg"
    endif
    if("$arg" == "-farm") then
	set FARMOUT
    endif
    if("$arg" == "-nocache") then
	set NOCACHE
    endif
end

if(! -e "$imgfile") then
    echo "usage: spotfinder.com whatever_001.img"
endif
if(! -e "$imgfile" && "$1" !~ *.img && $?XFORMSTATUSFILE) then
    set imgfile = `awk '{print $2}' $XFORMSTATUSFILE`
    if(-e "$imgfile") then
	echo "defaulting to $imgfile"
    endif
endif

if(! -e "$imgfile") then
    set BAD = "$imgfile does not exist"
    goto exit
endif

if($?FARMOUT && `hostname` !~ crush*) then
    set pwd = `pwd`
    if(-e /data/calibrations/crush_order.txt) then
	set rhost = `awk '{srand();for(i=1;i<=NF;++i)print rand(),$i}' /data/calibrations/crush_order.txt | sort -n | awk '{print $NF}' | head -1`
    else
	set rhosts = `ruptime -lr | awk '/crush/ && /up/ && $7<5{print $1}'`
	set rhost = $rhosts[1]
    endif
    if("$rhost" == "") then
	if( "$rhosts" =~ *crush14*) set rhost = crush14
    endif
    echo "farming out to $rhost"
    rsh -n $rhost "cd $pwd ; spotfinder.com $*"
    exit
endif

# decide on a place for temporary files
set tempfile = /tmp/${USER}/spotfinder_tmp$$
touch ${tempfile} >& /dev/null
if($status) then
    set tempfile = /tmp/spotfinder_tmp$$
    touch ${tempfile} >& /dev/null
    if($status) set tempfile = ~/spotfinder_tmp$$
endif

# make sure we remember the date
set imgdate  = `echo 'set time [file mtime "'$imgfile'"]; puts "$time [file size "'$imgfile'"] [clock format $time -format "%b %d %H:%M:%S %Y"]"' | tclsh`
# break this up so it means something
set epoch = $imgdate[1]
set size  = $imgdate[2]
set date  = "$imgdate[3] $imgdate[4] $imgdate[5] $imgdate[6]"
set year  = "$imgdate[6]"

# predict what the output files will be
set filename = `basename $imgfile`
set dir = `dirname $imgfile`
set dir = `cd $dir ; pwd`

# treat DVDs,etc as when they were local files
set test = `echo $dir | awk '{print substr($0,index($0,"/data"))}'`
if("$test" != "" && "$test" != "$dir") then
    set dir = "$test"
else
    set test = `echo $dir | awk '{print "/data/" substr($0,index($0,"/home/")+6)}'`
    if("$test" != "") set dir = "$test"
endif

# finalize new names
set imgname = "${dir}/${filename}"
set newlog = ${scoredir}/${year}${dir}/`basename $imgfile .img`_${epoch}.log
set newspt = ${scoredir}/${year}${dir}/`basename $imgfile .img`_${epoch}.spt

# check and see if this is redundant?
if(-e "$newlog" && -e "$newspt" && ! $?NOCACHE) then
    # skip the actual run if these already exist
    echo "cached: $newlog"
    echo "cached: $newspt"
    goto report
endif

# make a local copy of the image file (in case path is not writable)
cp $imgfile ${tempfile}.img
# do the run
spotfinder -k8 1 -ka 0 -kb 0 -kc 0 -kd 0 -ke 0 ${tempfile}.img >! ${tempfile}
if($status) then
    set BAD = "spotfinder failed to run"
endif
rm -f ${tempfile}.img >& /dev/null
set log = `awk '/Output/ && /.log$/{print $NF}' ${tempfile}`
set spt = `awk '/Output/ && /.spt$/{print $NF}' ${tempfile}`
if( ! -e "$log") then
    cat $tempfile
    set BAD = "failed to produce score log"
endif
if( ! -e "$spt") then
    cat $tempfile
    set BAD = "failed to produce spot file"
endif
rm -f ${tempfile} >& /dev/null
if($?BAD) goto exit

# move these to a more reasonable place
# build a tree under the "score" directory

# make sure the path exists first
mkdir -p `dirname $newspt`
echo "$imgname" >! $newlog
tail +2 $log >> $newlog
if("$status") then
    set BAD = "unable to create score log"
    goto exit
endif
echo "created: $newlog"
mv $spt $newspt
if("$status") then
    set BAD = "unable to create spot file"
    goto exit
endif
echo "created: $newspt"


echo "collected: $epoch $size $date" >> $newlog
# throw the header in there too
awk 'BEGIN{RS="\f"} {print;exit}' $imgfile >> $newlog

# make log have the same date as the image (so it's easier to keep track)
if(-e "$newlog") touch --reference=$imgfile $newlog
if(-e "$newspt") touch --reference=$imgfile $newspt

report:
if($?user_spt) then
    cp -p $newspt $user_spt
    echo "created $user_spt"
endif
set log = $newlog
# gather important numbers for a one-line summary of this run
set spotsize  = `awk '/spot size median/{print $1}' $log`
set spots     = `awk '/number of spots/{print $1;exit}' $log`
set overloads = `awk '/spots with overload/{print $1;exit}' $log`
set tooclose  = `awk '/number of spots with close neighbors/{print $1;exit}' $log`
set split     = `awk '/number of spots with multiple maxima/{print $1;exit}' $log`
set shape     = `awk '/spot shape median/{print $1}' $log`
set reso1     = `awk '/resolution boundary by Method 1/{reso=$1} /badness of curve in Method 1/{print reso"+/-"$1}' $log`
set reso2     = `awk '/resolution boundary by Method 2/{reso=$1} /badness of curve in Method 2/{print reso"+/-"$1}' $log`
set reso      = `awk '/average resolution boundary estimate/{reso=$1} /badness of curve in Method 1/{bad1=$1} /badness of curve in Method 2/{print reso"+/-"(($1+bad1)/2)} ' $log`
set icerings  = `awk '/number of ice-rings/{print $1}' $log`
set blasted   = `awk '/size of largest overloaded patch/{print $1}' $log`
set overlap   = `echo $split $tooclose | awk '{print $1+$2}'`

# extract the header info
set xsize    = `awk 'BEGIN{FS="="} /^SIZE1/{print $2+0}' $log | tail -1`
set ysize    = `awk 'BEGIN{FS="="} /^SIZE2/{print $2+0}' $log | tail -1`
set pixel    = `awk 'BEGIN{FS="="} /^PIXEL/{print $2+0}' $log | tail -1`
set xbeam    = `awk 'BEGIN{FS="="} /^BEAM_CENTER_X/{print $2+0}' $log | tail -1`
set ybeam    = `awk 'BEGIN{FS="="} /^BEAM_CENTER_Y/{print $2+0}' $log | tail -1`

set wave     = `awk 'BEGIN{FS="="} /^WAVELENGTH/{print $2+0}' $log | tail -1`
set dist     = `awk 'BEGIN{FS="="} /^DISTANCE/{print $2+0}' $log | tail -1`
set phi      = `awk 'BEGIN{FS="="} /^OSC_START/{print $2+0}' $log | tail -1`
set osc      = `awk 'BEGIN{FS="="} /^OSC_RANGE/{print $2+0}' $log | tail -1`
set exposure = `awk 'BEGIN{FS="="} /^TIME/{print $2+0}' $log | tail -1`

# make this all one line in the final output
if ("$reso1" == "") then
    set BAD = "unable to interpret spotfinder results"
    goto exit
endif

# make the one-line summary
echo -n "$epoch $size $date $imgname ${xsize}x${ysize} pixel=${pixel}mm "
echo -n "xbeam=${xbeam}mm ybeam=${ybeam}mm "
echo -n "exposure=${exposure}s wave=${wave}A dist=${dist}mm phi=$phi osc=$osc  "
echo -n "resolution=$reso  reso1=$reso1  reso2=$reso2 "
echo -n "$spots spots $overlap overlaps  $overloads overloads  "
echo    "shapescore=$shape  $icerings icerings"


exit:
if($?tempfile) then
    rm -f ${tempfile} >& /dev/null
endif

if($?BAD) then
    echo "ERROR: $BAD"
    exit 9
endif

exit



find /share/scores -name '*.log' -printf "%T@ %h %f\n" |\
awk '! ($3~/^107/ && length($3)==10) {base=substr($3,1,length($3)-4); \
  print "echo",$2"/"base \
  print "mv",$2"/"base".log",$2"/"$1"_"base".log"; \
  print "mv",$2"/"base".spt",$2"/"$1"_"base".spt"}' |\
cat >! sourceme


find /share/scores -type f -printf "%T@ %h %f\n" |\
awk '{name=substr($3,12,length($3)-15); print "mv",$2"/"$3,$2"/"name"_"$1 substr($3,length($3)-4)}' |\
tee sourceme

