/************************************************************************
                        Copyright 2003
                              by
                 The Board of Trustees of the 
               Leland Stanford Junior University
                      All rights reserved.
                       Disclaimer Notice
     The items furnished herewith were developed under the sponsorship
 of the U.S. Government.  Neither the U.S., nor the U.S. D.O.E., nor the
 Leland Stanford Junior University, nor their employees, makes any war-
 ranty, express or implied, or assumes any liability or responsibility
 for accuracy, completeness or usefulness of any information, apparatus,
 product or process disclosed, or represents that its use will not in-
 fringe privately-owned rights.  Mention of any product, its manufactur-
 er, or suppliers shall not, nor is it intended to, imply approval, dis-
 approval, or fitness for any particular use.  The U.S. and the Univer-
 sity at all times retain the right to use and disseminate the furnished
 items for any purpose whatsoever.                       Notice 91 02 01
   Work supported by the U.S. Department of Energy under contract
   DE-AC03-76SF00515; and the National Institutes of Health, National
   Center for Research Resources, grant 2P41RR01209. 
                       Permission Notice
 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:
 The above copyright notice and this permission notice shall be included
 in all copies or substantial portions of the Software.
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTA-
 BILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
 EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 THE USE OR OTHER DEALINGS IN THE SOFTWARE.
************************************************************************/


/*
 * The library consists of two files: libdistl.h, libdistl.cc.
 *
 * Help information is obtained by running the program 
 * with no command-line parameters.
 *
 * Developed by 
 *  Zepu Zhang,    zpzhang@stanford.edu
 *  Ashley Deacon, adeacon@slac.stanford.edu
 *  and others.
 *
 * July 2001 - May 2003.
 */


#include "libdistl.h"


diffimage::diffimage(): npxclassifyscan(3), nicecutoff(2)
{
	// Only processing parameters not bound to any specific image
	// are specified here.
	// Image properties are initialized when image data is initialized.
	
    overloadvalue = 65535;

    imgmargin = 20;

    scanboxsize[0] = 101;
    scanboxsize[1] = 51;
    scanboxsize[2] = 51;

    bgupperint[0] = 1.5;
    bgupperint[1] = 2.0;
    bgupperint[2] = 2.5;

    difflowerint = 3.8;

    iceringwidth = 4;
    iceresolmin = 1.0;
    iceresolmax = 8.0;

    icering_cutoffint[0] = 0.0;
    icering_cutoffint[1] = 1.5;

    icering_cutoffprct[0] = 0.55;
    icering_cutoffprct[1] = 0.20;

	icering_strength_cutweight[0] = 0.6;
	icering_strength_cutweight[1] = 0.4;
	// Weight of the two cutoff percentages in determining ice-ring strength.
	// Add to 1.
	// The lower cutoff (cutoffint[0]) measures continuity;
	// the higher cutoff (cutoffint[1]) measures intensity.

    spotarealowcut = 5;
	spotdistminfactor = 1.2;

    imgresolringpow = -3.0;
	imgresol_unispot_shiftfactor = 0.5;
	imgresol_unispace_cutoff_fraction = 0.15;
}


void diffimage::cleardata()
{
	for (int i = 0; i < pixelvalue.size(); i++)
		pixelvalue[i].clear();

	pixelvalue.clear();

	for (int i = 0; i < pixelintensity.size(); i++)
		pixelintensity[i].clear();

	pixelintensity.clear();

	for (int i = 0; i < pixellocalmean.size(); i++)
		pixellocalmean[i].clear();

	pixellocalmean.clear();

	maximas.clear();
	spots.clear();
	overloadpatches.clear();
	icerings.clear();

	imgresol_unispotresols.clear();
	imgresol_unispaceresols.clear();
	imgresol_unispacespotcounts.clear();
}


diffimage::~diffimage() 
{
	cleardata();

}


void diffimage::set_imageheader(const double pxsize, const double dist, const double wavelen,
		const double oscstart, const double oscrange, 
		const double beamctrx, const double beamctry) 
{
	// Clear data leftover from processing previous image.
	
	cleardata();

	// In setimageheader and setimagedata
	// eventually all image property values will be reset,
	// except for processing parameters.
	// This is to avoid problems while multiple files
	// are processed in a row.
	
	pixel_size = pxsize;
	distance = dist;
	wavelength = wavelen;
	resolb = pxsize * pxsize / dist / dist;
	osc_start = oscstart;
	osc_range = oscrange;
	beam_center_x = beamctrx;   
	beam_center_y = beamctry;  
	// The origin point wrt which the beam center is located is not clear.
	// Here it is assumed that beam_center_x and beam_center_y use
	// the same coord system as X and Y do, i.e.,
	// beam_center_x: top->bottom
	// beam_center_y: left->right

	beam_x = static_cast<int>(beam_center_x / pixel_size);	
	beam_y = static_cast<int>(beam_center_y / pixel_size);
}


void diffimage::set_imagedata(const int* const data, const int ncol, const int nrow)
{
	// DATA are stored by column, from left to right.
	// Type is int.

	ncols = ncol;
	nrows = nrow;

	pixelvalue.resize(ncols);
	pixelintensity.resize(ncols);
	pixellocalmean.resize(ncols);

	for (int x=0; x<ncols; x++) {
		pixelvalue[x] = vector<int>(data + x * nrows, data + (x+1) * nrows);
		pixelintensity[x].resize(nrows, 0);
		pixellocalmean[x].resize(nrows, 0);
	}

	underloadvalue = get_underload();
	
	firstx = imgmargin;
	lastx = ncols-1-imgmargin;
	firsty = imgmargin;
	lasty = nrows-1-imgmargin;
}



int diffimage::get_underload() const 
{
	// *****************************************************************
	// Determine the upperbound value for underloaded pixels, i.e.,
	// pixels with value <= UNDERLOAD are considered underloaded, like
	// those on the border or blocked by the beam stick.
	//
	// This function checks the whole image, not restricted by
	// 'firstx', 'lastx', 'firsty', 'lasty'.
	// *****************************************************************

	int ncols = pixelvalue.size();
	int nrows = pixelvalue[1].size();

	// Checks the pixels on the central cross of width 'corsswid'
	// and those on the four corners each of size 'barwidrow' by 'barwidcol'.

	int cornerfrac = 5;
	int barwidrow = nrows/cornerfrac;
	int barwidcol = ncols/cornerfrac;
	int crosswid = 40;

	int np = 4 * barwidrow * barwidcol + crosswid * (nrows + ncols) 
	   - crosswid*crosswid;
	vector<int> px(np);
	vector<int>::iterator p = px.begin();

	// Four corners.
	for (int icol=0; icol<barwidcol; icol++) {
		for (int irow=0; irow<barwidrow; irow++) {
			*p++ = pixelvalue[icol][irow];
			*p++ = pixelvalue[icol][nrows-irow-1];
			*p++ = pixelvalue[ncols-icol-1][irow];
			*p++ = pixelvalue[ncols-icol-1][nrows-irow-1];
		}
	}

	// Central cross.
	for (int icol=0; icol<ncols; icol++) {
		for (int irow=nrows/2-crosswid/2; irow<nrows/2+crosswid/2; irow++)
			*p++ = pixelvalue[icol][irow];
	}
	for (int icol=ncols/2-crosswid/2; icol<ncols/2+crosswid/2; icol++) {
		for (int irow=0; irow<nrows/2-crosswid/2; irow++)
			*p++ = pixelvalue[icol][irow];
		for (int irow=nrows/2+crosswid/2; irow<nrows; irow++)
			*p++ = pixelvalue[icol][irow];
	}


	// Check 'nq' percentiles of array 'px',
	// identify the value that is several consecutive
	// percentiles, which indicates that value happens 
	// a lot.

	int nq = 30;
	vector<int> q(nq);

	for (int i=0; i<nq; i++) {
		nth_element(px.begin()+i*np/nq, px.begin()+np*(i+1)/(nq+1), px.end());
		q[i] = px[np*(i+1)/(nq+1)];
	}

	int ul = q[nq/4]; 

	if (q[nq-1]==q[nq-2] && q[nq-2]==q[nq-3]) {
		nth_element(px.begin(), px.begin()+np*(nq+2)/(nq+3), px.end());
		ul = px[np*(nq+2)/(nq+3)];
	} else {
		for (int i=nq-2; i>1; i--)
			if (q[i]==q[i-1] && q[i-1]==q[i-2]) { 
				ul = q[i+1]; break; 
			}
	}


	return ul;
}


int diffimage::process()
{
	// The order of the following functions should not be changed 
	// without careful investigation.

printf("GOTHERE: pxlclassify()\n");
	pxlclassify();
printf("GOTHERE: search_icerings()\n");
	search_icerings();
printf("GOTHERE: search_maximas()\n");
	search_maximas();
printf("GOTHERE: search_spots()\n");
	search_spots();
printf("GOTHERE: search_overloadpatches()\n");
	search_overloadpatches();
printf("GOTHERE: imgresolution()\n");
	imgresolution();
printf("GOTHERE: done\n");

	return 0;
}


void diffimage::pxlclassify()
{
	// *********************************************
	// Classify pixels into underloaded, background,
	// diffracted, overloaded, etc.
	//
	// Scan the image box by box.
	// Some overlapping between boxes can be easily implemented, 
	// if needed.
	// **********************************************************


	for (int i=0; i<npxclassifyscan; i++) {
		if (scanboxsize[i] == 0) 
			continue;

		int xstart, xend, ystart, yend;

		int boxsize = scanboxsize[i];

		// First, scan the upper-left part. If any segments are left,
		// expand them to whole boxes and scan them.

		for (xstart=firstx; xstart<=lastx-boxsize+1; xstart+=boxsize-2) {
			xend = xstart + boxsize -1;
			for (ystart=firsty; ystart<=lasty-boxsize+1; ystart+=boxsize-2) {
				yend = ystart + boxsize - 1;
				pxlclassify_scanbox(xstart, xend, ystart, yend, 
					bgupperint[i]);      
			}
			if (yend<lasty)
				pxlclassify_scanbox(xstart, xend, lasty-boxsize+1, lasty, 
					bgupperint[i]);
		}

		if (xend<lastx) {
			xstart = lastx - boxsize + 1;
			xend = lastx;
			for (ystart=firsty; ystart<=lasty-boxsize+1; ystart+=boxsize-2) {
				yend = ystart + boxsize - 1;
				pxlclassify_scanbox(xstart, xend, ystart, yend, 
					bgupperint[i]); 
			}
			if (yend<lasty)
				pxlclassify_scanbox(xstart, xend, lasty-boxsize+1, lasty, 
					bgupperint[i]); 
		}

	}

}



void diffimage::pxlclassify_scanbox(const int xstart, const int xend, 
									const int ystart, const int yend,
                                    const double intensity_bguppercutoff) 
{
  // ******************************************************************************   
  // Calculate mean and std of all background pixels in the box.
  // Assign the calculated mean and std to each pixel in the box as its background.
  // Classify each pixel in the box based on this background.
  //
  // Expand the box if number of background pixels falls short of the required
  // amount.
  // ******************************************************************************

  int nbgmin = (xend - xstart + 1) * (yend - ystart + 1) * 2/3; 
  // Making 'nbgmin' reasonably small compared with the box size
  // is to avoid frequent need to expand the box due to shortage 
  // of background pixels.
  
  
  int boxnbg = 0;
  double boxmean = 0;
  double boxvar = 0;
  double boxstd = 0;

  for (int x=xstart; x<=xend; x++) {
    for (int y=ystart; y<=yend; y++) {
      int px = pixelvalue[x][y];
      if (pixelintensity[x][y]<intensity_bguppercutoff &&
	  px>underloadvalue &&
	  px<overloadvalue) {
	 
	boxmean += px;
	boxvar += static_cast<double>(px)*px;
	boxnbg++;
      }
    }
  }
  


  int xbox = xend - xstart + 1;
  int ybox = yend - ystart + 1;
  int halfincrease = min(xbox, ybox) / 3;
  int x0 = xstart;
  int y0 = ystart;

  while (boxnbg<nbgmin) {
    // Usually this loop is skipped because
    // usually boxnbg > nbgmin.

    xbox += 2*halfincrease;
    ybox += 2*halfincrease;
    x0 = min(max(0,x0-halfincrease), static_cast<int>(pixelvalue.size()-xbox));
    y0 = min(max(0,y0-halfincrease), static_cast<int>(pixelvalue[0].size()-ybox));
    // Whole image, rather than confined by firstx, firsty, lastx, lasty,
    // is used for scan.
    
    boxnbg = 0;
    boxmean = 0;
    boxvar = 0;

    for (int x=x0; x<x0+xbox; x++) {
      for (int y=y0; y<y0+ybox; y++) {
	int px = pixelvalue[x][y];
	if (pixelintensity[x][y]<intensity_bguppercutoff &&
	    px>underloadvalue &&
	    px<overloadvalue) {
	  boxmean += px;
	  boxvar += static_cast<double>(px)*px;
	  boxnbg++;
	}
      }
    }

  }
    
  
  boxmean = boxmean/boxnbg;
  boxvar = boxvar/boxnbg - boxmean*boxmean;
  boxstd = sqrt(boxvar);
  

  // Calculate pixel intensity.
  //
  // Intensity is calculated and assigned to the original box,
  // instead of the expansion, if any, to the box.

  double multfactor = 1.0 / boxstd;
  
  for (int x=xstart; x<=xend; x++) {
    for (int y=ystart; y<=yend; y++) {
      pixelintensity[x][y] = (pixelvalue[x][y] - boxmean) * multfactor;
	  pixellocalmean[x][y] = boxmean;
	  //pixellocalstd[x][y] = boxstd;
    }
  }


}



void diffimage::search_icerings()
{
	// *************************
	// Detect ice-rings.
	// *************************

	double icermin, icermax;

	double dx1 = beam_x - firstx;
	double dx2 = lastx - beam_x;
	double dy1 = beam_y - firsty;
	double dy2 = lasty - beam_y;
	dx1 = dx1 * dx1;
	dx2 = dx2 * dx2;
	dy1 = dy1 * dy1;
	dy2 = dy2 * dy2;
	icermax = max(max(dx1+dy1, dx1+dy2), max(dx2+dy1, dx2+dy2));
	double temp = resol_to_r2(iceresolmin);
	if (temp > icermax)
		icermax = sqrt(icermax);
	else
		icermax = sqrt(temp);

	icermin = sqrt(resol_to_r2(iceresolmax));

	// ring index increases as distance from ring to image center increases.
	// allocate one extra ring to avoid potential problems caused by weird (x,y) values.
	int nrings = static_cast<int>((icermax - icermin) / iceringwidth + 1);

	vector<int> ringnpx(nrings, 0);
	// Number of pixels on each ring.

	vector< vector<int> > ringcutoffnpx(nrings, vector<int>(nicecutoff, 0));
	// Number of pixels whose intensity exceeds the corresponding cutoff intensity values.


	// Sample every several pixels to save time.
	
	for (int x=firstx; x<=lastx; x = x + 2) {
		double dx = x - beam_x;
		for (int y=firsty; y<=lasty; y = y + 2) {
			double dy = y - beam_y;
			double r = sqrt(dx*dx + dy*dy);

			if (r > icermin && r < icermax) {
				int ringidx = static_cast<int>((r - icermin) / iceringwidth);
				if (ringidx < nrings) {
					ringnpx[ringidx] ++;
					int pixelint = pixelintensity[x][y];

					for (int cutoffidx = 0; cutoffidx < nicecutoff; cutoffidx++) 
						if (pixelint >= icering_cutoffint[cutoffidx])
							ringcutoffnpx[ringidx][cutoffidx] ++;
				}
			}
		}
	}




	///////////////////////////////////////////
	// Record the ice-rings found.
	// Contiguous ice-rings are combined to be counted as one.
	///////////////////////////////////////////

	icerings.reserve(5);

	int lasticering = -3;


	for (int ringidx = 0; ringidx < nrings; ringidx++) {
		if (ringnpx[ringidx] < 30)
			continue;


		bool passed = true;
		double thisstrength = 0.0;

		for (int cutoffidx=0; cutoffidx<nicecutoff; cutoffidx++) { 
			double prct = static_cast<double>(ringcutoffnpx[ringidx][cutoffidx]) / ringnpx[ringidx];

			if(prct < icering_cutoffprct[cutoffidx]) {
				// Specified intensity dosn't reach required fraction. -- not ice.
				passed = false;
				break;
			} else {
				thisstrength += (prct - icering_cutoffprct[cutoffidx]) / 
					(1.0 - icering_cutoffprct[cutoffidx]) * icering_strength_cutweight[cutoffidx];
			}
		}


		if (passed) {
			double r1 = icermin + iceringwidth*ringidx;
			double r2 = r1 + iceringwidth;
			if (ringidx > lasticering+1) {
				// a new ice-ring, not continuous with the previous one.
				icerings.push_back(icering());
				icerings.back().lowerr2 = r1*r1;
				icerings.back().upperr2 = r2*r2;
				icerings.back().upperresol = r2_to_resol(r1 * r1);
				icerings.back().lowerresol = r2_to_resol(r2 * r2);
				icerings.back().strength = thisstrength;
				icerings.back().npx = ringnpx[ringidx];

			} else {
				// expand the last ice-ring.
				icerings.back().upperr2 = r2*r2;
				icerings.back().lowerresol = r2_to_resol(r2 * r2);
				icerings.back().strength = max(icerings.back().strength, thisstrength);
				icerings.back().npx += ringnpx[ringidx];

			}

			lasticering = ringidx;
		}

	}

}



bool diffimage::pixelisonice(const int x, const int y) const
{
	// ******************************************************************
	// Check whether or not a pixel in on an ice-ring.
	// This function is called only when ice-ring does exist on the image.
	// ******************************************************************

	double resol = xy2resol(x, y);

	for (vector<icering>::const_iterator q=icerings.begin(); q!=icerings.end(); q++) {
		if (resol >= q->lowerresol) {
			if (resol <= q->upperresol)
				return (true);
			else
				return (false);
		}
	}

	return (false);
}



void diffimage::search_maximas()
{
	// ********************************
	// Search for local maximas.
	// ********************************


	// A local maxima is a pixel which is a diffraction pixel,
	// is not overloaded, and whose value is not smaller than
	// any of its 8 neighbors.

	int neighboramount = min(4, spotarealowcut-1);
	int diffnum = neighboramount - 1;

	for (int x=firstx+1; x<lastx; x++) {
		for (int y=firsty+1; y<lasty; y++) {

			if (pixelvalue[x][y] >= overloadvalue) {
				maximas.push_front(point(x, y, pixelvalue[x][y], pixelintensity[x][y]));

			} else {
				double pxint = pixelintensity[x][y];
				int pxv = pixelvalue[x][y];

				if (pxint > difflowerint &&
					pxv >= pixelvalue[x-1][y-1] &&
					pxv >= pixelvalue[x][y-1] &&
					pxv >= pixelvalue[x+1][y-1] &&
					pxv >= pixelvalue[x-1][y] &&
					pxv >= pixelvalue[x+1][y] &&
					pxv >= pixelvalue[x-1][y+1] &&
					pxv >= pixelvalue[x][y+1] &&
					pxv >= pixelvalue[x+1][y+1]) {
					int goodneighbors = 
						(pixelintensity[x-1][y-1] > difflowerint) +
						(pixelintensity[x][y-1] > difflowerint) +
						(pixelintensity[x+1][y-1] > difflowerint) +
						(pixelintensity[x-1][y] > difflowerint) +
						(pixelintensity[x+1][y] > difflowerint) +
						(pixelintensity[x-1][y+1] > difflowerint) +
						(pixelintensity[x][y+1] > difflowerint) +
						(pixelintensity[x+1][y+1] > difflowerint); 
					if (goodneighbors > diffnum) { 
						maximas.push_front(point(x, y, pixelvalue[x][y], pixelintensity[x][y]));
					}
				}
			}
		}
	}


}


void diffimage::search_spots()
{
	// ************************************************
	// Search for spots. 
	// Record area and calculate intensity of each spot.
	//
	// Every maxima belongs to a unique spot; but
	// one spot could have multiple maximas, depending
	// on whether neighborhoods of two maximas are
	// connnected, in which case both maximas are
	// enclosed in one single spot.
	// ************************************************

	const double PI = 3.14159265;
	
	int nx = pixelvalue.size();
	int ny = pixelvalue[0].size();
	vector< vector<bool> > pixelvisited(nx, vector<bool>(ny, false));

	for (list<point>::iterator p=maximas.begin(); p!=maximas.end(); p++) {
		if (pixelvisited[p->x][p->y]) 
			continue;

		//if (pixelvalue[p->x][p->y] >= overloadvalue)
		//	continue;

		spots.push_back(spot());
		search_border_spot(p->x, p->y, spots.back(), pixelvisited);


		// The following filtering is done here in order to ease 
		// the burden on memory.
		
		// Discard this spot if smaller than specified threshold.
		if (spots.back().bodypixels.size() < spotarealowcut)
		{
			spots.pop_back();
		}
		else
		{ 
			// Discard this spot if any of its pixel lies on an ice-ring.
			for (list<point>::const_iterator q=spots.back().borderpixels.begin();
					q!=spots.back().borderpixels.end(); q++) {
				if (pixelisonice(q->x, q->y)) {
					spots.pop_back();
					break;
				}
			}
		}
	}


	if (spots.empty())
		return;

	
	// Calculate spot properties.
	
	vector< vector<bool> > pixelismaxima(nx, vector<bool>(ny, false));
	for (list<point>::const_iterator p=maximas.begin(); p!=maximas.end(); p++)
		pixelismaxima[p->x][p->y] = true;


	for (list<spot>::iterator p=spots.begin(); p!=spots.end(); p++) 
	{
		//p->maximas.clear();  // unnecessary
		for (list<point>::const_iterator q=p->bodypixels.begin();
		     q!=p->bodypixels.end(); q++) {
			if (pixelismaxima[q->x][q->y]) {
			  p->maximas.push_back( *q );

			  if (p->maximas.size() == 1)  
			    p->peak = *q;
			  else 
			    if (q->value > p->peak.value)
			      p->peak = *q;
			}
		}


		if (p->maximas.size() > 1)
		// If all maximas are close together, view it as a single-maxima spot.
		{
			double avgx = 0;
			double avgy = 0;

			for (list<point>::const_iterator q = p->maximas.begin();
				q != p->maximas.end(); q++)
			{
				avgx += q->x;
				avgy += q->y;
			}

			avgx = avgx / p->maximas.size();
			avgy = avgy / p->maximas.size();

			double maxxdist = 0;
			double maxydist = 0;

			for (list<point>::const_iterator q = p->maximas.begin();
				q != p->maximas.end(); q++)
			{
				maxxdist = max(maxxdist, abs(avgx - q->x));
				maxydist = max(maxydist, abs(avgy - q->y));
			}

			if (maxxdist < 2 & maxydist < 2)
			{
				int newx = static_cast<int>(avgx);
				int newy = static_cast<int>(avgy);
				p->peak = point(newx, newy, pixelvalue[newx][newy], pixelintensity[newx][newy]);
				p->maximas.clear();
				p->maximas.push_back(p->peak);
			}
		}



		p->peakresol = xy2resol(p->peak.x, p->peak.y);


		// Calculate spot center, weighted by body pixel value.

		p->centerx = 0.0;
		p->centery = 0.0;
		double pxv;
		double pxvsum = 0.0;
		for (list<point>::const_iterator q=p->bodypixels.begin(); q!=p->bodypixels.end(); q++)
			{
			pxv = static_cast<double>(pixelvalue[q->x][q->y]);
			p->centerx += q->x * pxv;
			p->centery += q->y * pxv;
			pxvsum += pxv;
			}

		p->centerx /= pxvsum;
		p->centery /= pxvsum;
		p->pxvsum  = pxvsum;


		p->ncloseneighbors = 0;


		// Calculate spot shape.
		// There're better algorithms for mean and var.

		vector<double> borderdist;
		borderdist.reserve(p->borderpixels.size());
		double meanborderdist = 0;

		for (list<point>::const_iterator q=p->borderpixels.begin(); q!=p->borderpixels.end(); q++)
		{	
			double dist = sqrt(static_cast<double>(q->x - p->centerx)*(q->x - p->centerx) 
						+ static_cast<double>(q->y - p->centery)*(q->y - p->centery));
			borderdist.push_back(dist);
			meanborderdist += dist;
		}
		meanborderdist /= p->borderpixels.size();

		double varborderdist = 0;
		for (vector<double>::const_iterator q=borderdist.begin(); q!=borderdist.end(); q++)
			varborderdist += (*q - meanborderdist) * (*q - meanborderdist);
		varborderdist /= p->borderpixels.size() - 1;

		p->shape = sqrt(varborderdist) / meanborderdist;

		// Trhansform the above value to (1, 0) from (0.12, 0.25), 
		// corresponding to bounds of usual values.
		// slope = 1.0 / (0.12 - 0.25) = -7.69;
		// intercept = -slope * 0.25 = 1.92.
		p->shape = -7.69 * p->shape + 1.92;
	}
	

	search_neighbor_spots();

}





void diffimage::search_border_spot(const int x, const int y, spot& curspot, 
                                   vector< vector<bool> >& pixelvisited)
{
  // **********************************************************
  // Searches for contiguous diffraction pixels to be included
  // in one spot.
  // ************************************************************


  if (pixelvisited[x][y]) 
    return;

  if (x<firstx || x>lastx || y<firsty || y>lasty)
    // out of image border
    return;


  pixelvisited[x][y] = true;

  if (pixelintensity[x][y]<=difflowerint) {
    // Hit a nighboring pixel, which does not belong to 
    // this spot. Record border pixels and return.
  	
    curspot.borderpixels.push_front(point(x, y, pixelvalue[x][y], pixelintensity[x][y]));

    // From this step it is clear that a border pixel
    // is one that is bordering a spot, but does not
    // belong to the spot, i.e, is not counted in the
    // spot area.
    // Perimeter is number of such border pixels.
  } else {
    // An interior pixel on the spot.
    // Label current pixel and search on.
 
    curspot.bodypixels.push_front(point(x, y, pixelvalue[x][y], pixelintensity[x][y]));

  
    //search_border_spot(x-1, y-1, curspot, pixelvisited);
    search_border_spot(x,   y-1, curspot, pixelvisited);
    //search_border_spot(x+1, y-1, curspot, pixelvisited);
    search_border_spot(x-1, y,   curspot, pixelvisited);
    search_border_spot(x+1, y,   curspot, pixelvisited);
    //search_border_spot(x-1, y+1, curspot, pixelvisited);
    search_border_spot(x  , y+1, curspot, pixelvisited);
    //search_border_spot(x+1, y+1, curspot, pixelvisited);
  }
  
}


void diffimage::search_overloadpatches()
{
	int nx = pixelvalue.size();
	int ny = pixelvalue[0].size();
	vector< vector<bool> > pixelvisited(nx, vector<bool>(ny, false));

	for (list<point>::iterator p=maximas.begin(); p!=maximas.end(); p++) {
		if (!pixelvisited[p->x][p->y]) {
			if (pixelvalue[p->x][p->y] >= overloadvalue) {
				// If maxima is overloaded, search for a overloaded patch.
				overloadpatches.push_back(spot());
				search_border_overload(p->x, p->y, overloadpatches.back(), pixelvisited);

				// Remove this maxima from maxima list.
				list<point>::iterator pp = p;
				p--;
				maximas.erase(pp);
			} 
		}
	}
}


void diffimage::search_border_overload(const int x, const int y, spot& curspot, 
                                   vector< vector<bool> >& pixelvisited)
{
	// One possible undesirable situation is that a spot is found 
	// bordering an overload patch. 
	// But it is unlikely to happen often.

  if (pixelvisited[x][y])
    return;

  if (x<firstx || x>lastx || y<firsty || y>lasty)
    // out of image border
    return;

  pixelvisited[x][y] = true;

  if (pixelvalue[x][y]<overloadvalue) {
    // Hit a nighboring pixel, which does not belong to 
    // this spot. Record border pixels and return.
   
    curspot.borderpixels.push_front(point(x, y, pixelvalue[x][y], pixelintensity[x][y]));

    // From this step it is clear that a border pixel
    // is one that is bordering a spot, but does not
    // belong to the spot, i.e, is not counted in the
    // spot area.
    // Perimeter is number of such border pixels.
  } else {
    // An interior pixel on the spot.
    // Label current pixel and search on.
 
    curspot.bodypixels.push_front(point(x, y, pixelvalue[x][y], pixelintensity[x][y]));

    search_border_overload(x-1, y-1, curspot, pixelvisited);
    search_border_overload(x,   y-1, curspot, pixelvisited);
    search_border_overload(x+1, y-1, curspot, pixelvisited);
    search_border_overload(x-1, y,   curspot, pixelvisited);
    search_border_overload(x+1, y,   curspot, pixelvisited);
    search_border_overload(x-1, y+1, curspot, pixelvisited);
    search_border_overload(x  , y+1, curspot, pixelvisited);
    search_border_overload(x+1, y+1, curspot, pixelvisited);
  }
}




void diffimage::search_neighbor_spots()
{

	vector<double> spotperimeter;
	spotperimeter.reserve(spots.size());

	for (list<spot>::const_iterator p = spots.begin(); p != spots.end(); p++)
	{
		//spotaxis.push_back( p->majoraxis );
		spotperimeter.push_back( p->borderpixels.size() );
	}

	int upperidx = static_cast<int>( (spots.size() - 1) * 0.98 );
	nth_element(spotperimeter.begin(), spotperimeter.begin() + upperidx, spotperimeter.end());
	double upperperimeter = spotperimeter[upperidx];
	double mindist = upperperimeter / 3.1416 * spotdistminfactor;


	// Sort spots in increasing order of spot's X location.
	spots.sort( cmpspotx() );

	
	list<spot>::iterator p = spots.begin();
	int pidx = 0;

	for (; pidx < spots.size() - 1; pidx++) {
		list<spot>::iterator q = p;
		int qidx = pidx + 1;

		for (; qidx < spots.size(); qidx++) {
			q++;

			// Since q is closest to p in X direction,
			// if they're far apart in X direction,
			// no other spot will come closer.
			// In this case, p has no close neighbor.
			
			if (q->peak.x - p->peak.x > mindist)
				break;

			// q and p are close in X direction but far apart
			// in Y direction.
			// There could be spots that are slightly farther
			// in X direction but close in Y direction.
			// Check them out.
			
			if (q->peak.y - p->peak.y > mindist ||
				p->peak.y - q->peak.y > mindist)
				continue;

			// q and p are close enough.
			// Both are marked to be deleted.
			// Continue to check other neighbors.
			double dist = (q->peak.x - p->peak.x) * (q->peak.x - p->peak.x) +
				(q->peak.y - p->peak.y) * (q->peak.y - p->peak.y);
			//double localmindist = max(p->majoraxis, q->majoraxis) * 2.0 * spotdistminfactor;
			double localmindist = max(p->borderpixels.size(), q->borderpixels.size()) / 
				3.14 * spotdistminfactor;
			localmindist = localmindist * localmindist;

			if (dist <= localmindist)
			{
				p->ncloseneighbors ++;
				q->ncloseneighbors ++;
			}
		}

		p++;
	}

}



void diffimage::imgresolution()
{
	const double PI = 3.14159265;

	imgresol_unispot = 99.0;
	imgresol_unispace = 99.0;
	imgresol_unispot_curvebadness = 1.0;
	imgresol_unispace_curvebadness = 1.0;

	/////////////////////////////////////////
	// Calulate corner modifiers
	// to be used to adjust amount of spots.
	/////////////////////////////////////////


	// Resolution and corner factor of each spot.
	vector<double> spotresol;
	vector<int> spotcornerfactor;
	spotresol.reserve(spots.size());
	spotcornerfactor.reserve(spots.size());

	// Largest radius with complete circle on the image
	int commonr = min(min(beam_x-firstx, lastx-beam_x), 
					min(beam_y-firsty, lasty-beam_y));
	double rupper = beam_x - firstx;
	double rlower = lastx - beam_x;
	double rleft = beam_y - firsty;
	double rright = lasty - beam_y;

	// Ignore spots with 2 * theta < 2.9 degrees.
	const double resolupperbnd = twotheta_to_resol(2.9 / 180.0 * PI);


	for (list<spot>::const_iterator p = spots.begin();
			p != spots.end(); p++) {

		// Don't consider spots that are 
		//    overloaded or
		//    2 * theta < 2.9 degreed, i.e., too close to beam center.
		if (p->peak.value >= overloadvalue || p->peakresol > resolupperbnd)
		{
			continue;
		}
			
	
		spotresol.push_back(p->peakresol);

		double dx = p->peak.x - beam_x;
		double dy = p->peak.y - beam_y;
		double r = sqrt(dx*dx + dy*dy);
		if (r < commonr) {
			spotcornerfactor.push_back(1);
		} else {
			double goodang = 2.0 * PI;

			if (r > rupper)
			goodang -= 2.0 * acos(rupper/r);
			if (r > rright)
			goodang -= 2.0 * acos(rright/r);
			if (r > rlower)
			goodang -= 2.0 * acos(rlower/r);
			if (r > rleft)
			goodang -= 2.0 * acos(rleft/r);

			if (goodang < 1e-2)
				goodang = 1e-2;

			spotcornerfactor.push_back(static_cast<int>(2.0*PI/goodang));
		}
	}	


	// Order spots from low to high resolution
	sort(spotresol.begin(), spotresol.end(), greater<double>());




	// Duplicate spots whose cornerfactor is greater than 1.
	int nspots = accumulate(spotcornerfactor.begin(),spotcornerfactor.end(),0);

	// If there are spots in the corners,
	// copy corner spots to make total number of spots the expected amount.
	// This insertion does not change the fact that the vector is ordered.
	if (nspots > spotcornerfactor.size()) {
		spotresol.reserve(nspots);
		for (int sptidx=spotresol.size()-1; sptidx>=0; sptidx--) {
			if (spotcornerfactor[sptidx]>1)
				spotresol.insert(spotresol.begin()+sptidx,
						spotcornerfactor[sptidx]-1,spotresol[sptidx]);
		}
	}


	int nimgresolrings;
printf("GOTHERE: imgresolution() Method 1\n");


	//
	// Curve for Method 1:
	//   evenly divide number of spots, check trend of resolution
	//
	
printf("GOTHERE: imgresolution() nimgresolrings\n");
	nimgresolrings = min(40, static_cast<int>(spotresol.size() / 25));
	if(nimgresolrings == 0.0) nimgresolrings = 2.0 ;
printf("GOTHERE: imgresolution() nspotring\n");
printf("GOTHERE: imgresolution() nimgresolrings = %f\n", nimgresolrings);
printf("GOTHERE: imgresolution() spotresol.size() = %f\n", spotresol.size());
	int nspotring = spotresol.size() / nimgresolrings;

	// resolution at the outer border of each ring. 
printf("GOTHERE: imgresolution() imgresol_unispotresols\n");
	imgresol_unispotresols = vector<double>(nimgresolrings); 
printf("GOTHERE: imgresolution() imgresol_unispotresolspow\n");
	vector<double> imgresol_unispotresolspow(nimgresolrings);

printf("GOTHERE: imgresolution() ringidx loop\n");
	for (int ringidx=0; ringidx<nimgresolrings; ringidx++) {
		int idx = (ringidx + 1) * nspotring - 1;
		imgresol_unispotresols[ringidx] = spotresol[idx];
		imgresol_unispotresolspow[ringidx] = pow(imgresol_unispotresols[ringidx], imgresolringpow);
	}


	/*
	vector<double> ringresolpowsmooth(imgresol_unispotresolspow.size());

	vector<double> ringindices(imgresol_unispotresolspow.size());
	for (int ringidx=0; ringidx<imgresol_unispotresolspow.size(); ringidx++)
		ringindices[ringidx] = static_cast<double>(ringidx);
	ksmooth(ringindices, ringresolpow, ringindices, ringresolpowsmooth, smoothspan);
	*/



printf("GOTHERE: imgresolution() slope of line\n");
	// Slope of line connecting 1st and each other ring.
	vector<double> slope(imgresol_unispotresolspow.size());
	slope[0] = 0;
	for (int ringidx=1; ringidx<imgresol_unispotresolspow.size(); ringidx++) 
		slope[ringidx] = (imgresol_unispotresolspow[ringidx] - 
				imgresol_unispotresolspow[0]) / ringidx;


printf("GOTHERE: imgresolution() Method 1 badness\n");
	// Badness of the curve in Method 1
	// The spots should be more and more sparse as one moves from beam center outwards.
	// Badness of the curve measures violation of this trend.


	// Number of possible pairwise comparisons.
	int ntotalcomb = (slope.size() - 1) * (slope.size() - 2) / 2;
	if (slope.size() > 1 && ntotalcomb > 0) {
		int nbadcomb = 0;

		for (int ringidx = 1; ringidx < slope.size()-1; ringidx++) {
			for (int idx = ringidx+1; idx < slope.size(); idx++) {
				if (slope[idx] <= slope[ringidx])
					nbadcomb ++;
			}
		}

		imgresol_unispot_curvebadness = static_cast<double>(nbadcomb) / ntotalcomb;
	}



printf("GOTHERE: imgresolution() Method 2\n");
	//
	// Curve for Method 2:
	//   evenly divide reciprocal resolution space, check number of spots 
	//
	

	double resolpowmin = pow(spotresol[0] + 0.1, imgresolringpow);
	
	// Plus a little for convenience in the following process.
	// No need to worry about overflow.
	double resolpowmax = pow(spotresol.back() + 0.1, imgresolringpow);

	int ringspotcount0 = max(25, static_cast<int>(spotresol.size() / 20));

	double resolpowstep = pow(spotresol[ringspotcount0 - 1], imgresolringpow) 
		- resolpowmin;

	nimgresolrings = max( 8, min(30, static_cast<int>((resolpowmax - resolpowmin) / resolpowstep)) );

	resolpowstep = (resolpowmax - resolpowmin) / nimgresolrings;


	imgresol_unispaceresols = vector<double>(nimgresolrings, 0.0);
	for (int ringidx = 0; ringidx < nimgresolrings; ringidx++) {
		imgresol_unispaceresols[ringidx] = pow( resolpowmin + (ringidx + 1) * resolpowstep, 
				1.0/imgresolringpow );
	}

	imgresol_unispacespotcounts = vector<int>(nimgresolrings, 0);
	int curspotidx = 0;


	// Notice that spotresol is ordered from large to small.
	for (int ringidx = 0; ringidx < nimgresolrings; ringidx++) {
		while (spotresol[curspotidx] >= imgresol_unispaceresols[ringidx]) {
			imgresol_unispacespotcounts[ringidx] ++;
			curspotidx ++;
		}

	}



printf("GOTHERE: imgresolution() Method 2 badness\n");
	// Badness of the curve in Method 2
	// Number of spots should decrease as one moves from beam center outwards.
	// Badness of the curve measures violation of this trend.

	// Number of possible pairwise comparisons.
	ntotalcomb = imgresol_unispacespotcounts.size() * (imgresol_unispacespotcounts.size() - 1) / 2;
	if (ntotalcomb > 0) {
		int nbadcomb = 0;
		for (int ringidx=1; ringidx < imgresol_unispacespotcounts.size()-1; ringidx++) {
			for (int idx=ringidx+1; idx<imgresol_unispacespotcounts.size(); idx++) {
				if (imgresol_unispacespotcounts[idx] >= imgresol_unispacespotcounts[ringidx])
					nbadcomb ++;
			}
		}

		imgresol_unispace_curvebadness = static_cast<double>(nbadcomb) / ntotalcomb;
	}



	// If there're fewer than 25 spots lower than 4 Angstrom.
	
	if (spotresol.size() <= 25 || spotresol[24] < 4.0) {
		return;
	}




printf("GOTHERE: imgresolution() Method 1 execute\n");
	// Method 1

	
	if (slope.size() > 4) {

		int maxslopeidx = 0;
		for (int ringidx=1; ringidx<slope.size(); ringidx++) {
			if (slope[ringidx] > slope[maxslopeidx])
				maxslopeidx = ringidx;
		}


		//int maxslopeidx = distance(slope.begin(), max_element(slope.begin(),slope.end()));

		double a = slope[maxslopeidx];
		double b = imgresol_unispotresolspow[0];
		vector<double> gap(maxslopeidx+1);
		int bendidx = 0;
		for (int ringidx=0; ringidx<=maxslopeidx; ringidx++) {
			gap[ringidx] = a*ringidx+b - imgresol_unispotresolspow[ringidx];
			if (gap[ringidx] > gap[bendidx])
				bendidx = ringidx;
		}
		//int bendidx = distance(gap.begin(), max_element(gap.begin(),gap.end()));

		double temp1 = 0;
		double temp2 = 0;
		for (int ringidx=0; ringidx<=maxslopeidx; ringidx++) {
			temp1 += gap[ringidx];
			temp2 += gap[ringidx] * gap[ringidx];
		}
		temp1 /= (maxslopeidx + 1.0);
		temp2 /= (maxslopeidx + 1.0);
		double gapstd = sqrt(temp2 - temp1*temp1);


		double gapmax = gap[bendidx];
		double gapcutoff = gapmax - imgresol_unispot_shiftfactor * gapstd;

		for (int ringidx=bendidx; ringidx<=maxslopeidx; ringidx++) {
			if (gap[ringidx] >= gapcutoff)
				bendidx = ringidx;
			else
				break;
		}

		imgresol_unispot = imgresol_unispotresols[bendidx];
	}



printf("GOTHERE: imgresolution() Method 2 execute\n");
	//
	// Method 2:
	//


	int cutofffrac = imgresol_unispacespotcounts[0] * 0.5 + imgresol_unispacespotcounts[1] * 0.5; 
	cutofffrac *= imgresol_unispace_cutoff_fraction;

	if (cutofffrac >= 2) {

		// Walk through bins until spot count drops below the threshold.

		for (int ringidx = 2; ringidx < imgresol_unispacespotcounts.size() - 1; ringidx++) {
			if ( (imgresol_unispacespotcounts[ringidx] < cutofffrac &&
				imgresol_unispacespotcounts[ringidx+1] < cutofffrac) ||
				ringidx == imgresol_unispacespotcounts.size() - 2) {
				imgresol_unispace = pow(resolpowmin + resolpowstep * ringidx, 1.0/imgresolringpow);
				break;
			}
		}
	}

}



//inline double diffimage::r2_to_resol(const double r2) const
double diffimage::r2_to_resol(const double r2) const
{
  // ********************************************
  // Determine resolution from squared radius.
  // This function is the reverse of resol_to_r2.
  // ********************************************

  // Calculation of resolution:
  //
  // d = lambda / (2*sin(theta))
  // sin(theta) = sqrt( 1/2 - 1/(2 * sqrt(1+(h/L)^2)) )
  // lambda: wavelength
  // h: distance from spot to beam center
  //    h^2 = ( (x-beam_x)^2 + (y-beam_y)^2 ) * pixel_size^2
  // L: distance from detector to crystal
  // ====>
  //    d = lambda / sqrt( 2 - 2/sqrt(1 + h^2/L^2) )
  //      = wavelength / sqrt( 2 - 2/sqrt(1 + pixel_size^2/L^2 * 
  //                      ( (x-beam_x)^2 + (y-beam_y)^2 ) ) )
  //    b = pixel_size^2 / L^2
  //    c = (x-beam_x)^2 + (y-beam_y)^2
  //    d = sqrt(1 + c * b)
  //    two_sin_theta = sqrt(2 - 2/d)
  //    resolution = wavelength / two_sin_theta

  // resol = wavelength / sqrt(2 - 2/sqrt(1 + b*r^2) )
  // wavelength on the order of 1, b on the order of 1e-7.

  // for b = 1.6e-7, wavelength = 1,
  // 1/d^3 is approx proportional to r^2.8

  // d = wavelength / sqrt(2 - 2/(1 + b*r*r))
  // 1 / sqrt(1 + b*r*r) approx. 1 + b*r*r*(-1/2)
  // ==> d approx. wavelength/sqrt(b*r*r) = A / r
  // So resolution is approx. proportional to 1/r,
  // where r is distance from pixel to beam center.


      
  //static double resolb = pow(pixel_size / distance, 2);

  double d = sqrt(1.0 + resolb * r2);
  double two_sin_theta = sqrt(2.0 - 2.0/d);

  if (two_sin_theta<10e-8)
    return 10e8;
  else
    return wavelength/two_sin_theta;

}


//inline double diffimage::xy2resol(const double x, const double y) const
double diffimage::xy2resol(const double x, const double y) const
{
	double r2 = (x - beam_x)*(x - beam_x) + (y - beam_y)*(y - beam_y);
	return r2_to_resol(r2);
}



double diffimage::resol_to_r2(const double resol) const 
{
  // ********************************************
  // Determine squared radius from resolution.
  // This function is the reverse of r2_to_resol.
  // ********************************************

  // static double b = pow(pixel_size / distance, 2);

  double sin_theta, cos_2theta, tan_2theta_sqr;

  sin_theta = 0.5 * wavelength / resol;
  // When this function is called properly, 
  // 'resol' should be reasonably larger than 0.

  double d = 1.0 / (1.0 - 2 * sin_theta * sin_theta);

  return (d * d - 1.0) / resolb;
}


//inline double diffimage::twotheta_to_resol(const double twotheta) const
double diffimage::twotheta_to_resol(const double twotheta) const
{
	return wavelength / 2.0 / sin(twotheta * 0.5);
}



// The following utility functions are not used.
/*

template<class T1, class T2> 
int ksmooth(const vector<T1>& X, const vector<T2>& Y, const vector<T1>& fitloc, vector<T2>& fit, const double span)
{
	//
	// Kernel smoother.
	//

	// 'X' are independents.
	// 'Y' are dependents corresponding to 'X'. 
	// 'fitloc' passs in predictors, e.g., a copy of X.
	// 'fit' returns predictions.
	// 'span' specifies search neighborhood to be span multiplied by range of 'X',
	// that is, span/2 * range(X) on each side of a predictor.  0 < span < 1.
	// Triangle kernel is used.
	//
	// Prediction is a weighted average of variables in the neighborhood of the predictor:
	//   fit = r(i) * Y(i) / SUM( r(i) ) 
	// where 
	//       r(i) = min(0, 1 - (|X(i) - predictor|)/(span/2 * range(X)))

	double hspan = (*max_element(X.begin(),X.end()) - *min_element(X.begin(),X.end())) * span * 0.5;

	for (int j=0; j<fitloc.size(); j++) {
		double r;
		double sumr = 0;
		double prediction = 0;

		for (int i=0; i<X.size(); i++) {
			r = abs(X[i] - fitloc[j]);
			if (r < hspan) {
				r = 1.0 - r/hspan;
				prediction += r * Y[i];  
				sumr += r;
			}
		}

		fit[j] = static_cast<T2>(prediction / sumr);
	}

	return 0;
}


*/

