/* convert ideal pixel intensities into noisy pixels						-James Holton 		5-5-13

example:

gcc -O -o noisify noisify.c -lm

./noisify -bin floatimage.bin -distance 100 -detsize 100 -pixel 0.1 \
  -scale 1 -readout 3 -flicker 0.02 -calibration 0.03

wavelength (lambda) should be provided in Angstrom
detector distance, detsize and pixel size in mm
the -scale value is multiplied by every value found in floatimage.bin before use

floatimage.bin should be a binary "dumpfile" consisting of the proper number of 4-byte
"float" numbers on the current architecture.  These numbers should be in "photons/pixel" scale.
The nearBragg and fastBragg programs can be used to generate it.

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#define PI 3.14159265358979
#define twoPI 6.28318530717959



/* generate unit vector in random direction */
float uniform3Ddev(float *dx, float *dy, float *dz, long *idum);
/* random deviate with Poisson distribution */
float poidev(float xm, long *idum);
/* random deviate with Gaussian distribution */
float gaussdev(long *idum);
/* random deviate with Lorentzian distribution */
float lorentzdev(long *idum);
/* random deviate with triangle-shaped distribution */
float triangledev(long *idum);
/* random deviate with exponential distribution (>0) */
float expdev(long *idum);
/* random deviate with uniform distribution */
float ran1(long *idum);


char *floatfilename = "floatimage.bin\0";
FILE *floatfile = NULL;
char *headerfilename = "header.img\0";
FILE *headerfile = NULL;
char header[2048];
char *intfilename = "intimage.img\0";
char *noisefilename = "noiseimage.img\0";
FILE *outfile = NULL;

int main(int argc, char** argv)
{

    /* detector parameters used to make the header */
    /* assumed to be the same as those used to call nearBragg/fastBragg!  */

    double distance = 100.0e-3;
    double detsize_x = 102.4e-3;
    double detsize_y = 102.4e-3;
    double pixel = 0.1e-3;
    double Xdet,Ydet,Xbeam=-1e99,Ybeam=-1e99,Rdet;
    int xpixel,ypixel,xpixels=0,ypixels=0,pixels;
    double lambda = 1;
       
    int n,i,j;
    float *floatimage;
    unsigned short int *intimage;

    double test,sum,photons,photons0,adu;
    double readout_noise=0.0, flicker_noise=0.0;
    double calibration_noise=0.03;
    double adc_offset = 40.0;
    double quantum_gain = 1.0;
    int overloads = 0;

    double phi0 = 0, osc = 1;
                
    /* Thomson cross section */
    double r_e_sqr = 7.94079248018965e-30;
    /* incident x-ray fluence in photons/m^2   default equivalent to unity
  	that is, one electron will scatter 1 ph/SR after a fluence of 1.26e29 ph/m^2
        this places the input file on a photons/pixel scale */
    double fluence = 1.25932015286227e+29;
    /* arbitrary "photon scale" applied before calculating noise, default is unity */
    double photon_scale = 1.0;
    double scale;

    double I;
    double max_I = 0.0;
        
    long seed;
    long calib_seed = 123456789;
        
    seed = -time((time_t *)0);
//    printf("GOTHERE seed = %u\n",seed);
      
        
    /* check argument list */
    for(i=1; i<argc; ++i)
    {
        if(argv[i][0] == '-')
        {
            /* option specified */
            if(strstr(argv[i], "-lambda") && (argc >= (i+1)))
            {
		/* copy directly into image header */
                lambda = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-Xbeam") && (argc >= (i+1)))
            {
                Xbeam = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-Ybeam") && (argc >= (i+1)))
            {
                Ybeam = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-distance") && (argc >= (i+1)))
            {
                distance = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-detsize") && (strlen(argv[i]) == 8) && (argc >= (i+1)))
            {
                detsize_x = atof(argv[i+1])/1000.0;
                detsize_y = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-detsize_x") && (argc >= (i+1)))
            {
                detsize_x = atof(argv[i+1])/1000.0;
            }
             if(strstr(argv[i], "-detsize_y") && (argc >= (i+1)))
            {
                detsize_y = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-detpixels") && (strlen(argv[i]) == 10) && (argc >= (i+1)))
            {
                xpixels = ypixels = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-detpixels_x") && (argc >= (i+1)))
            {
                xpixels = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-detpixels_y") && (argc >= (i+1)))
            {
                ypixels = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-pixel") && (argc >= (i+1)))
            {
                pixel = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-fluence") && (argc >= (i+1)))
            {
                fluence = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-floatfile") && (argc >= (i+1)))
            {
                floatfilename = argv[i+1];
		floatfile = fopen(floatfilename,"r");
            }
            if(strstr(argv[i], "-header") && (argc >= (i+1)))
            {
                headerfilename = argv[i+1];
		headerfile = fopen(headerfilename,"r");
		fread(header,1,512,headerfile);
            }
            if(strstr(argv[i], "-intfile") && (argc >= (i+1)))
            {
                intfilename = argv[i+1];
            }
            if(strstr(argv[i], "-noisefile") && (argc >= (i+1)))
            {
                noisefilename = argv[i+1];
            }
            if(strstr(argv[i], "-readout") && (argc >= (i+1)))
            {
                readout_noise = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-flicker") && (argc >= (i+1)))
            {
                flicker_noise = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-calibration") && (argc >= (i+1)))
            {
                calibration_noise = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-scale") && (argc >= (i+1)))
            {
                photon_scale = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-seed") && (argc >= (i+1)))
            {
                seed = -atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-calib_seed") && (argc >= (i+1)))
            {
                calib_seed = -atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-adc") && (argc >= (i+1)))
            {
                adc_offset = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-gain") && (argc >= (i+1)))
            {
                quantum_gain = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-phi") && (argc >= (i+1)))
            {
                phi0 = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-osc") && (argc >= (i+1)))
            {
                osc = atof(argv[i+1]);
            }
        }
    }

    if(floatfile == NULL){
	printf("usage: noisify -floatfile floatimage.bin\n");
	printf("options:\n");\
	printf("\tfloatimage.bin\t nearBragg-style binary dump file\n");
        printf("\t-scale\tscale factor to put floatimage.bin in photons/pixel\n");
        printf("\t-gain\tpixel units per photon\n");
        printf("\t-readout_noise\tgaussian noise added to every pixel\n");
        printf("\t-flicker\t fractional 1/f noise in source\n");
        printf("\t-calibration\t static fractional error per pixel\n");
        printf("\t-calib_seed\t change seed for calibration error\n");
        printf("\t-seed\t specify seed for all non-calibration errors\n");
        printf("\t-gain\t pixel units per photon\n");
        printf("\t-adc\t offset added to each pixel after noise\n");
        printf("\t-distance\t distance from origin to detector center in mm\n");
        printf("\t-detsize\t detector size in mm\n");
        printf("\t-pixel\t detector pixel size in mm\n");
        printf("\t-lambda\t incident x-ray wavelength in Angstrom\n");
        printf("\t-floatfile\t name of binary output file (4-byte floats)\n");
        printf("\t-intfile\t name of smv-formatted output file (arbitrary scale)\n");
        printf("\t-noisefile\t name of smv-formatted output file (with Poisson noise)\n");
        printf("\t-Xbeam\t image X coordinate of direct-beam spot (mm)\n");
        printf("\t-Ybeam\t image Y coordinate of direct-beam spot (mm)\n");
        printf("\t-header\t import 512-byte header from specified SMV file\n");
exit(9);
    }

    
    /* allocate memory */
    if(xpixels) {
	detsize_x = pixel*xpixels;
    }
    if(ypixels) {
	detsize_y = pixel*ypixels;
    }
    xpixels = ceil(detsize_x/pixel);
    ypixels = ceil(detsize_y/pixel);
    pixels = xpixels*ypixels;
    floatimage = calloc(pixels+10,sizeof(float));
    intimage = calloc(pixels+10,sizeof(unsigned short int));

    printf("importing intensity: %s\n",floatfilename);
    fread(floatimage,pixels,sizeof(float),floatfile);
    fclose(floatfile);

    /* defaults? */
    if(Xbeam <= -1e99) Xbeam = detsize_x/2.0;
    if(Ybeam <= -1e99) Ybeam = detsize_y/2.0;  


    if(headerfile != NULL)
    {
	printf("taking header from %s\n",headerfilename);
    }
    else
    {
        printf("  distance=%g detsize=%gx%g  pixel=%g meters (%dx%d pixels)\n",distance,detsize_x,detsize_y,pixel,xpixels,ypixels);
        printf("  Xbeam=%g Ybeam=%g\n",Xbeam,Ybeam);
    }
    printf("  seed: %ld\n",seed);
    printf("  calibration noise seed: %ld\n",calib_seed);
    printf("  calibration_noise = %g %%\n",calibration_noise*100);
    printf("  input file scale = %g\n",photon_scale);
    printf("  readout_noise = %g ADU\n",readout_noise);
    printf("  flicker_noise = %g %%\n",flicker_noise*100);
    printf("  quantum_gain = %g ADU/photon\n",quantum_gain);
    printf("  adc_offset = %g ADU\n",adc_offset);
 

    printf("\n");

    /* output as ints */   
    j = 0;
    max_I = 0.0;
    for(j=0;j<pixels;++j){
	/* optionally scale the input file */
	I = photon_scale*floatimage[j];
	/* convert into photons/pixel (no change unless user specified fluence) */
	photons = I*fluence*r_e_sqr;
	/* convert photons/pixel into area detector units */
	adu = photons*quantum_gain+adc_offset;
	if(max_I < I) max_I = I;
	if(I < 0.0) printf("WARNING: negative intensity in %s: %g\n",floatfilename,I);
	if(adu > 65535.0) adu = 65535.0;
	intimage[j] = (unsigned short int) ( adu );
//	printf("%d %d = %d\n",xpixel,ypixel,intimage[j]);
    }
    printf("maximum intensity in file: %g\n",max_I);

    printf("writing %s as %lu-byte integers\n",intfilename,sizeof(unsigned short int));
    outfile = fopen(intfilename,"w");
    if(headerfile != NULL)
    {
	fwrite(header,1,512,outfile);
    }
    else
    {
        fprintf(outfile,"{\nHEADER_BYTES=512;\nDIM=2;\nBYTE_ORDER=little_endian;\nTYPE=unsigned_short;\n");
        fprintf(outfile,"SIZE1=%d;\nSIZE2=%d;\nPIXEL_SIZE=%g;\nDISTANCE=%g;\n",xpixels,ypixels,pixel*1000.0,distance*1000.0);
        fprintf(outfile,"WAVELENGTH=%g;\nBEAM_CENTER_X=%g;\nBEAM_CENTER_Y=%g;\n",lambda,Xbeam*1000.0,(detsize_y-Ybeam)*1000);
        fprintf(outfile,"PHI=%g;\nOSC_START=%g;\nOSC_RANGE=%g;\n",phi0,phi0,osc);
        fprintf(outfile,"DETECTOR_SN=000;\n");
        fprintf(outfile,"BEAMLINE=fake;\n");
        fprintf(outfile,"}\f");
        while ( ftell(outfile) < 512 ){ fprintf(outfile," "); };
    }
    fwrite(intimage,sizeof(unsigned short int),pixels,outfile);
    fclose(outfile);



    /* simulate noise */
    j = 0;
    sum = 0.0;
    overloads = 0;
    for(j=0;j<pixels;++j){

	/* ideal photons/pixel */
	photons0 = floatimage[j] * photon_scale * fluence * r_e_sqr;

	/* simulate 1/f noise in source */
	if(flicker_noise > 0.0){
	    photons0 *= ( 1.0 + flicker_noise * gaussdev( &seed ) );
        }
	/* calibration is same from shot to shot, so use different seed */
	if(calibration_noise > 0.0){
	    photons0 *= ( 1.0 + calibration_noise * gaussdev( &calib_seed ) );
        }
	photons = poidev( photons0, &seed );

	/* accumulate number of photons */
	sum += photons;
	/* convert photons to pixel units */
	adu = photons*quantum_gain;
	adu += adc_offset;

	/* readout noise is in pixel units? */
	if(readout_noise > 0.0){
	    adu += readout_noise * gaussdev( &seed );
        }

	if(adu > 65535.0) {
	    adu = 65535.0;
	    ++overloads;
	}
	intimage[j] = (unsigned short int) adu;
//	printf("pixel %d = %d\n",j,intimage[j]);
    }
    printf("%.0f photons on noise image (%d overloads)\n",sum,overloads);

    printf("writing %s as %lu-byte integers\n",noisefilename,sizeof(unsigned short int));
    outfile = fopen(noisefilename,"w");
    if(headerfile != NULL)
    {
	fwrite(header,1,512,outfile);
    }
    else
    {
        fprintf(outfile,"{\nHEADER_BYTES=512;\nDIM=2;\nBYTE_ORDER=little_endian;\nTYPE=unsigned_short;\n");
        fprintf(outfile,"SIZE1=%d;\nSIZE2=%d;\nPIXEL_SIZE=%g;\nDISTANCE=%g;\n",xpixels,ypixels,pixel*1000.0,distance*1000.0);
        fprintf(outfile,"WAVELENGTH=%g;\nBEAM_CENTER_X=%g;\nBEAM_CENTER_Y=%g;\n",lambda,Xbeam*1000.0,(detsize_y-Ybeam)*1000);
        fprintf(outfile,"PHI=%g;\nOSC_START=%g;\nOSC_RANGE=%g;\n",phi0,phi0,osc);
        fprintf(outfile,"DETECTOR_SN=000;\n");
        fprintf(outfile,"BEAMLINE=fake;\n");
        fprintf(outfile,"}\f");
        while ( ftell(outfile) < 512 ){ fprintf(outfile," "); };
    }
    fwrite(intimage,sizeof(unsigned short int),pixels,outfile);
    fclose(outfile);

    return;
}



/* returns a unit vector in a random direction in arguments dx,dy,dz */
/* also returns a random magnitude within the unit sphere as a return value */
float uniform3Ddev(float *dx, float *dy, float *dz, long *seed)
{
    float ran1(long *idum);
    float dr;
    
    /* pick a random direction by cutting a sphere out of a cube */
    dr = 0;
    while(dr>1 || dr < 1e-2)
    {
        *dx = 2.1*(ran1(seed)-0.5);
        *dy = 2.1*(ran1(seed)-0.5);
        *dz = 2.1*(ran1(seed)-0.5);
        dr = sqrt(*dx**dx+*dy**dy+*dz**dz);
    }
    /* turn this into a unit vector */
    *dx/=dr;
    *dy/=dr;
    *dz/=dr;
    
    /* dx,dy,dz should now be a random unit vector */
    
    return dr;
}


float poidev(float xm, long *idum)
{
    float gammln(float xx);
    float ran1(long *idum);
    /* oldm is a flag for whether xm has changed since last call */
    static float sq,alxm,g,oldm=(-1.0);
    float em,t,y;
    
    /* routine below locks up for > 1e6 photons? */
    if (xm > 1.0e6) {
	return xm+sqrt(xm)*gaussdev(idum);
    }

    if (xm < 12.0) {
        /* use direct method: simulate exponential delays between events */
        if(xm != oldm) {
            /* xm is new, compute the exponential */
            oldm=xm;
            g=exp(-xm);
        }
        /* adding exponential deviates is equivalent to multiplying uniform deviates */
        /* final comparison is to the pre-computed exponential */
        em = -1;
        t = 1.0;
        do {
            ++em;
            t *= ran1(idum);
        } while (t > g);
    } else {
        /* Use rejection method */
        if(xm != oldm) {
            /* xm has changed, pre-compute a few things... */
            oldm=xm;
            sq=sqrt(2.0*xm);
            alxm=log(xm);
            g=xm*alxm-gammln(xm+1.0);
        }
        do {
            do {
                /* y is a deviate from a lorentzian comparison function */
                y=tan(PI*ran1(idum));
                /* shift and scale */
                em=sq*y+xm;
            } while (em < 0.0);		/* there are no negative Poisson deviates */
            /* round off to nearest integer */
            em=floor(em);
            /* ratio of Poisson distribution to comparison function */
            /* scale it back by 0.9 to make sure t is never > 1.0 */
            t=0.9*(1.0+y*y)*exp(em*alxm-gammln(em+1.0)-g);
        } while (ran1(idum) > t);
    }
        
    return em;
}


/* return gaussian deviate with rms=1 and FWHM = 2/sqrt(log(2)) */
float gaussdev(long *idum)
{
    float ran1(long *idum);
    static int iset=0;
    static float gset;
    float fac,rsq,v1,v2;
        
    if (iset == 0) {
        /* no extra deviats handy ... */
        
        /* so pick two uniform deviates on [-1:1] */
        do {
            v1=2.0*ran1(idum)-1.0;
            v2=2.0*ran1(idum)-1.0;
            rsq=v1*v1+v2*v2;
        } while (rsq >= 1.0 || rsq == 0);
        /* restrained to the unit circle */
        
        /* apply Box-Muller transformation to convert to a normal deviate */
        fac=sqrt(-2.0*log(rsq)/rsq);
        gset=v1*fac;
        iset=1;		/* we now have a spare deviate */
        return v2*fac;
    } else {
        /* there is an extra deviate in gset */
        iset=0;
        return gset;
    }
}


/* generate Lorentzian deviate with FWHM = 2 */
float lorentzdev(long *seed) {
    float ran1(long *idum);
 
    return tan(M_PI*(ran1(seed)-0.5));
}

/* return triangular deviate with FWHM = 1 */
float triangledev(long *seed) {
    float ran1(long *idum);
    float value;

    value = ran1(seed);
    if(value > 0.5){
        value = sqrt(2*(value-0.5))-1;
    }else{
        value = 1-sqrt(2*value);
    }

    return value;
}



float expdev(long *idum)
{
    float dum;
    
    do
    dum=ran1(idum);
    while( dum == 0.0);
    return -log(dum);
}



/* ln of the gamma function */
float gammln(float xx)
{
    double x,y,tmp,ser;
    static double cof[6]={76.18009172947146,-86.50532032941677,
    24.01409824083091,-1.231739572450155,
    0.1208650973866179e-2,-0.5395239384953e-5};
    int j;
    
    y=x=xx;
    tmp=x+5.5;
    tmp -= (x+0.5)*log(tmp);
    ser = 1.000000000190015;
    for(j=0;j<=5;++j) ser += cof[j]/++y;
    
    return -tmp+log(2.5066282746310005*ser/x);
}





/* returns a uniform random deviate between 0 and 1 */
#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define EPS 1.2e-7
#define RNMX (1.0-EPS)

float ran1(long *idum)
{
    int j;
    long k;
    static long iy=0;
    static long iv[NTAB];
    float temp;
    
    if (*idum <= 0 || !iy) {
        /* first time around.  don't want idum=0 */
        if(-(*idum) < 1) *idum=1;
        else *idum = -(*idum);
        
        /* load the shuffle table */
        for(j=NTAB+7;j>=0;j--) {
            k=(*idum)/IQ;
            *idum=IA*(*idum-k*IQ)-IR*k;
            if(*idum < 0) *idum += IM;
            if(j < NTAB) iv[j] = *idum;
        }
        iy=iv[0];
    }
    /* always start here after initializing */
    k=(*idum)/IQ;
    *idum=IA*(*idum-k*IQ)-IR*k;
    if (*idum < 0) *idum += IM;
    j=iy/NDIV;
    iy=iv[j];
    iv[j] = *idum;
    if((temp=AM*iy) > RNMX) return RNMX;
    else return temp;
}

