/* perfect-lattice nanocrystal diffraction simulator						-James Holton and Ken
Frankel		9-30-13

example:

gcc -O -o fastBragg fastBragg.c -lm

./fastBragg -mat auto.mat -hkl P1.hkl -distance 2500

./fastBragg -mat A.mat -hkl P1.hkl -lambda 1 -dispersion 0.1 -dispstep 3 -distance 100  -detsize 100 -pixel 0.1 \
  -hdiv 0.28 -hdivstep 0.02 -vdiv 0.28 -vdivstep 0.02 \
  -fluence 1e24 -N 0 \
  -water 0

lattice positions and wavelength (lambda) should be provided in Angstrom, three numbers per line
detector distance, detsize and pixel size in mm
divergence in mrad
dispersion in percent
phi and osc are in degrees
fluence is in photons/meter^2 (integrated exposure time)
Na, Nb, Nc, are the number of unit cells along the a,b,c axes, respectively
    note that any of Na,Nb,Nc can be zero to simulate an isolated unit cell (SAXS)
water is the thickness in microns of "water" also traversed by the beam
    this generates a simplitic background: that from a material with density 1.0 and isotropic 
    structure factor of 2.57 electrons (the forward-scattered structure factor of water
    more complicated backgrounds can be made in a separate run of this program using Na=Nb=Nc=0.

auto.mat can be an orientation matrix from MOSFLM, or simply a text file of the
three reciprocal lattice vector components along x,y,z:
a_star_x b_star_x c_star_x
a_star_y b_star_y c_star_y
a_star_z b_star_z c_star_z

P1.hkl should be a text file containing
h k l F
for EVERY spot that has an intensity (including F000).  No symmetry operators will
be imposed by this program.  Not even Friedel symmetry.

Since reading the HKL file can often be the slowest step, this program will create
a binary "dumpfile" in the current working directory that it will re-read upon 
subsequent runs if -hkl is not specified.

Please note that unlike nearBragg, this program does not work in the near field,
so detector distances should always be much larger than the crystal size

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#define PI 3.14159265358979
#define twoPI 6.28318530717959


/*  interpolation functions */
void polint(double *xa, double *ya, double x, double *y);
void polin2(double *x1a, double *x2a, double **ya, double x1,double x2, double *y);
void polin3(double *x1a, double *x2a, double *x3a, double ***ya, double x1,double x2, double x3, double *y);



/* rotate a 3-vector in space applied in order phix,phiy,phiz*/
double *rotate(double *v, double *new, double phix, double phiy, double phiz);
/* rotate a 3-vector about a unit vector axis */
double *rotate_axis(double *v, double *new, double *axis, double phi);

/* generate unit vector in random direction */
float uniform3Ddev(float *dx, float *dy, float *dz, long *idum);
/* random deviate with Poisson distribution */
float poidev(float xm, long *idum);
/* random deviate with Gaussian distribution */
float gaussdev(long *idum);
/* random deviate with Lorentzian distribution */
float lorentzdev(long *idum);
/* random deviate with triangle-shaped distribution */
float triangledev(long *idum);
/* random deviate with exponential distribution (>0) */
float expdev(long *idum);
/* random deviate with uniform distribution */
float ran1(long *idum);

/* Fourier transform of a truncated lattice */
double sincg(double x, double N);
/* Fourier transform of a sphere */
double sinc3(double x);
/* Fourier transform of a spherically-truncated lattice */
double sinc_conv_sinc3(double x);


char *matfilename;
FILE *matfile = NULL;
char *hklfilename;
FILE *hklfile = NULL;
char *dumpfilename = "Fdump.bin\0";
FILE *dumpfile = NULL;
char *floatfilename = "floatimage.bin\0";
char *sinfilename = "sinimage.bin\0";
char *cosfilename = "cosimage.bin\0";
char *intfilename = "intimage.img\0";
char *noisefilename = "noiseimage.img\0";
FILE *outfile = NULL;

int main(int argc, char** argv)
{
    int printout = 0;
    int printout_ypixel,printout_xpixel=-1;
    int accumulate = 0;
    double r,dmin=0;

    /* Thomson cross section */
    double r_e_sqr = 7.94079248018965e-30;
    /* incident x-ray fluence in photons/m^2 */
    double fluence = 1.25932015286227e+29;
    /* polarization factor */
    double polar,costwotheta;


    /* water background stuff */
    double I_water;
    double water_size = 0.0;
    double water_scale = 1.75;
    // 2.57^2*r_e_sqr*1e6*6.022e23/18.0


    double distance = 100.0e-3;
    double detsize_x = 102.4e-3;
    double detsize_y = 102.4e-3;
    double xdet_vector[4]  = {0,0,0,1};
    double ydet_vector[4]  = {0,0,-1,0};
    double zdet_vector[4]  = {0,1,0,0};
    double pix0_vector[4]  = {0,0,0,0};
    double beam_vector[4]  = {0,1,0,0};
    double polar_vector[4] = {0,0,0,1};
    double spindle_vector[4] = {0,0,0,1};
    int curved_detector = 0;
    double pixel = 0.1e-3;
    double subpixel;
    double Xdet,Ydet,Xbeam=-1e99,Ybeam=-1e99,Rdet;
    int xpixel,ypixel,xpixels=0,ypixels=0,pixels;
    long progress_pixel,progress_pixels;
    int progress_meter=1;
    int calculate_noise = 1;
    int babble=1;
    int oversample = -1,suby,subz;
    double xtalsize_max,xtalsize_a,xtalsize_b,xtalsize_c;
    int roi_xmin=-1,roi_xmax=-1,roi_ymin=-1,roi_ymax=-1;
    double airpath,omega_pixel,omega_Rsqr_pixel;
    int point_pixel = 0;
    int nopolar = 0;

    double stol,twotheta,theta;

    double lambda,dispersion=0.0,dispstep=-1,lambda0 = 1.0e-10;
    double phi,phi0=0.0,phistep=-1.0,osc=-1.0;
    double hdiv,hdivstep=-1.0,hdivrange= -1.0;
    double vdiv,vdivstep=-1.0,vdivrange= -1.0;
        
    double source_path,source_distance = 10.0;
    double detector_twotheta = -1e99;
        
    int n;
    int steps,divsteps=-1,hdivsteps=-1,vdivsteps=-1,dispsteps=-1,phisteps=-1;
    int hdiv_tic,vdiv_tic,phi_tic,disp_tic;
    double d_hdiv,d_vdiv,d_disp,dd_hdiv,dd_vdiv,dd_disp;

    int Fs=1000000;
    double F_latt,F_cell;
    double denom;

    /* interpolation arrays */
    int interpolate = 2;
    double ***Fhkl;
    double ***sub_Fhkl;
    int    h_interp[5],k_interp[5],l_interp[5];
    double h_interp_d[5],k_interp_d[5],l_interp_d[5];

    double h,k,l;
    int    h0,k0,l0,h_range,k_range,l_range,h_min,h_max,k_min,k_max,l_min,l_max;
    int    h0_flr,k0_flr,l0_flr;
    int    i1=0, i2=0, i3=0;


    int    N=1;
    double Na=1.0,Nb=1.0,Nc=1.0;
    int    round_xtal = 0;
    int    round_div = 1;
    int    binary_spots = 0;
    double a_x,a_y,a_z,b_x,b_y,b_z,c_x,c_y,c_z;
    double a_x0,a_y0,a_z0,b_x0,b_y0,b_z0,c_x0,c_y0,c_z0;
    double a,b,c,alpha,beta,gamma;
    double a_star,b_star,c_star,a_star_x,a_star_y,a_star_z,b_star_x,b_star_y,b_star_z,c_star_x,c_star_y,c_star_z;
    double alpha_star,beta_star,gamma_star;
    double a_cross_b_x,a_cross_b_y,a_cross_b_z;
    double b_cross_c_x,b_cross_c_y,b_cross_c_z;
    double c_cross_a_x,c_cross_a_y,c_cross_a_z;
    double a_star_cross_b_star_x,a_star_cross_b_star_y,a_star_cross_b_star_z;
    double b_star_cross_c_star_x,b_star_cross_c_star_y,b_star_cross_c_star_z;
    double c_star_cross_a_star_x,c_star_cross_a_star_y,c_star_cross_a_star_z;
    double V_cell,V_star,RTD=180.0/PI;
    double sinalpha,sinbeta,singamma,sinalpha_star,sinbeta_star,singamma_star;

    int i,j;
    float *floatimage;
    float *sinimage;
    float *cosimage;
    unsigned short int *intimage;
    double X,Y,Z,X0,Y0,Z0,d_r,diffray_x,diffray_y,diffray_z,xd,yd,zd,xd0,yd0,zd0,Xdet0,Ydet0;

    double test,sum,intfile_scale = 0;
    int overloads = 0;
        
    /* x-ray beams */
    double vector[4];
    double newvector[4];
    double pixel_X,pixel_Y,pixel_Z;
    double source_X,source_Y,source_Z;

    /* scattering vectors */
    double S0_x,S0_y,S0_z,S0;
    double S_x,S_y,S_z,S;
    double s_x,s_y,s_z,s;

    double phase_path,phase,Fa,Fb,I;
    double max_I = 0.0;
    double max_I_x,max_I_y;
    int coherent = 0;
    int integral_form = 0;
        
    long seed;
        
    seed = -time((time_t *)0);
      
        
    /* check argument list */
    for(i=1; i<argc; ++i)
    {
        if(argv[i][0] == '-')
        {
            /* option specified */
            if(strstr(argv[i], "-Na") && (argc >= (i+1)))
            {
                Na = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-Nb") && (argc >= (i+1)))
            {
                Nb = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-Nc") && (argc >= (i+1)))
            {
                Nc = atoi(argv[i+1]);
            }
            if(0==strcmp(argv[i], "-N") && (argc >= (i+1)))
            {
                Na = Nb = Nc = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-Xbeam") && (argc >= (i+1)))
            {
                Xbeam = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-Ybeam") && (argc >= (i+1)))
            {
                Ybeam = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-xdet_vector") && (argc >= (i+3)))
            {
                xdet_vector[1] = atof(argv[i+1]);
                xdet_vector[2] = atof(argv[i+2]);
                xdet_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-ydet_vector") && (argc >= (i+3)))
            {
                ydet_vector[1] = atof(argv[i+1]);
                ydet_vector[2] = atof(argv[i+2]);
                ydet_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-zdet_vector") && (argc >= (i+3)))
            {
                zdet_vector[1] = atof(argv[i+1]);
                zdet_vector[2] = atof(argv[i+2]);
                zdet_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-beam_vector") && (argc >= (i+3)))
            {
                beam_vector[1] = atof(argv[i+1]);
                beam_vector[2] = atof(argv[i+2]);
                beam_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-polar_vector") && (argc >= (i+3)))
            {
                polar_vector[1] = atof(argv[i+1]);
                polar_vector[2] = atof(argv[i+2]);
                polar_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-spindle_axis") && (argc >= (i+3)))
            {
                spindle_vector[1] = atof(argv[i+1]);
                spindle_vector[2] = atof(argv[i+2]);
                spindle_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-pix0_vector") && (argc >= (i+3)))
            {
                pix0_vector[0] = 1.0;
                pix0_vector[1] = atof(argv[i+1]);
                pix0_vector[2] = atof(argv[i+2]);
                pix0_vector[3] = atof(argv[i+3]);
            }
            if(strstr(argv[i], "-divergence") && (argc >= (i+1)))
            {
                hdivrange = vdivrange = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-hdivrange") && (argc >= (i+1)))
            {
                hdivrange = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-vdivrange") && (argc >= (i+1)))
            {
                vdivrange = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-divergence") && (argc >= (i+1)))
            {
                hdivrange = vdivrange = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-hdivstep") && (strlen(argv[i]) == 9) && (argc >= (i+1)))
            {
                hdivstep = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-hdivsteps") && (argc >= (i+1)))
            {
                hdivsteps = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-vdivstep") && (strlen(argv[i]) == 9) && (argc >= (i+1)))
            {
                vdivstep = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-vdivsteps") && (argc >= (i+1)))
            {
                vdivsteps = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-divsteps") && (argc >= (i+1)))
            {
                hdivsteps = vdivsteps = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-distance") && (argc >= (i+1)))
            {
                distance = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-source_distance") && (argc >= (i+1)))
            {
		source_distance = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-twotheta") && (argc >= (i+1)))
            {
                detector_twotheta = atof(argv[i+1])/RTD;
            }
            if(strstr(argv[i], "-detsize") && (strlen(argv[i]) == 8) && (argc >= (i+1)))
            {
                detsize_x = atof(argv[i+1])/1000.0;
                detsize_y = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-detsize_x") && (argc >= (i+1)))
            {
                detsize_x = atof(argv[i+1])/1000.0;
            }
             if(strstr(argv[i], "-detsize_y") && (argc >= (i+1)))
            {
                detsize_y = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-detpixels") && (strlen(argv[i]) == 10) && (argc >= (i+1)))
            {
                xpixels = ypixels = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-detpixels_x") && (argc >= (i+1)))
            {
                xpixels = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-detpixels_y") && (argc >= (i+1)))
            {
                ypixels = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-curved_det") && (argc >= (i+1)))
            {
                curved_detector = 1;
            }
            if(strstr(argv[i], "-pixel") && (argc >= (i+1)))
            {
                pixel = atof(argv[i+1])/1000.0;
            }
            if(strstr(argv[i], "-point_pixel") )
            {
                point_pixel = 1;
            }
            if(strstr(argv[i], "-nopolar") )
            {
                nopolar = 1;
            }
            if(strstr(argv[i], "-oversample") && (argc >= (i+1)))
            {
                oversample = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-roi") && (argc >= (i+4)))
            {
                roi_xmin = atoi(argv[i+1]);
                roi_xmax = atoi(argv[i+2]);
                roi_ymin = atoi(argv[i+3]);
                roi_ymax = atoi(argv[i+4]);
            }
            if((strstr(argv[i], "-lambda") || strstr(argv[i], "-wave")) && (argc >= (i+1)))
            {
                lambda0 = atof(argv[i+1])/1.0e10;
            }
            if(strstr(argv[i], "-energy") && (argc >= (i+1)))
            {
                lambda0 = (12398.42/atof(argv[i+1]))/1.0e10;
            }
            if(strstr(argv[i], "-fluence") && (argc >= (i+1)))
            {
                fluence = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-dispersion") && (argc >= (i+1)))
            {
                dispersion = atof(argv[i+1])/100.0;
            }
            if(strstr(argv[i], "-dispsteps") && (argc >= (i+1)))
            {
                dispsteps = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-phi") && strlen(argv[i])==4 && (argc >= (i+1)))
            {
                phi0 = atof(argv[i+1])/RTD;
            }
            if(strstr(argv[i], "-osc") && (argc >= (i+1)))
            {
                osc = atof(argv[i+1])/RTD;
            }
            if(strstr(argv[i], "-phistep") && strlen(argv[i])==8 && (argc >= (i+1)))
            {
                phistep = atof(argv[i+1])/RTD;
            }
            if(strstr(argv[i], "-phisteps") && (argc >= (i+1)))
            {
                phisteps = atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-mat") && (argc >= (i+1)))
            {
                matfilename = argv[i+1];
		matfile = fopen(matfilename,"r");
            }
            if(strstr(argv[i], "-hkl") && (argc >= (i+1)))
            {
                hklfilename = argv[i+1];
		hklfile = fopen(hklfilename,"r");
            }
            if(strstr(argv[i], "-dmin") && (argc >= (i+1)))
            {
                dmin = atof(argv[i+1])*1e-10;
            }
            if((strstr(argv[i], "-floatfile") || strstr(argv[i], "-floatimage")) && (argc >= (i+1)))
            {
                floatfilename = argv[i+1];
            }
            if((strstr(argv[i], "-intfile") || strstr(argv[i], "-intimage")) && (argc >= (i+1)))
            {
                intfilename = argv[i+1];
            }
            if((strstr(argv[i], "-noisefile") || strstr(argv[i], "-noiseimage")) && (argc >= (i+1)))
            {
                noisefilename = argv[i+1];
		calculate_noise = 1;
            }
            if(strstr(argv[i], "-nonoise") )
            {
                /* turn off noise */
                calculate_noise = 0;
            }
            if(strstr(argv[i], "-scale") && (argc >= (i+1)))
            {
		/* specify the scale for the intfile */
                intfile_scale = atof(argv[i+1]);
            }
            if(strstr(argv[i], "-coherent") )
            {
		/* turn off incoherent addition */
                coherent = 1;
            }
            if(strstr(argv[i], "-printout") )
            {
		/* turn on console printing */
                printout = 1;
            }
            if(strstr(argv[i], "-noprogress") )
            {
		/* turn off progress meter */
                progress_meter = 0;
            }
            if(strstr(argv[i], "-progress") )
            {
		/* turn off progress meter */
                progress_meter = 1;
            }
            if(strstr(argv[i], "-interpolate") )
            {
		/* turn on tricubic interpolation */
                interpolate = 1;
            }
            if(strstr(argv[i], "-nointerpolate") )
            {
		/* turn off tricubic interpolation */
                interpolate = 0;
            }
            if(strstr(argv[i], "-round_xtal") )
            {
		/* use sinc3 */
                round_xtal = 1;
            }
            if(strstr(argv[i], "-square_xtal") )
            {
		/* use sincg */
                round_xtal = 0;
            }
            if(strstr(argv[i], "-round_div") )
            {
		/* cut to circle */
                round_div = 1;
            }
            if(strstr(argv[i], "-square_div") )
            {
		/* just raster */
                round_div = 0;
            }
            if(strstr(argv[i], "-binary_spots") )
            {
		/* cut off everything at FWHM */
                binary_spots = 1;
            }
            if(strstr(argv[i], "-printout_pixel") && (argc >= (i+2)))
            {
                printout_xpixel = atoi(argv[i+1]);
                printout_ypixel = atoi(argv[i+2]);
            }
            if(strstr(argv[i], "-seed") && (argc >= (i+1)))
            {
                seed = -atoi(argv[i+1]);
            }
            if(strstr(argv[i], "-water") && (argc >= (i+1)))
            {
                water_size = atof(argv[i+1])/1e6;
            }
        }
    }

    printf("fastBragg diffraction simulator - James Holton and Ken Frankel 5-10-13\n");

    if(matfile == NULL){
	printf("usage: fastBragg -mat auto.mat -hkl Fs.hkl\n");
	printf("options:\n");\
	printf("\t-mat filename.mat\tmosflm-style matrix file containing three reciprocal unit cell vectors\n");
	printf("\t-hkl filename.hkl\ttext file containing h, k, l and F for P1 unit cell\n");
        printf("\t-distance        \tdistance from origin to detector center in mm\n");
        printf("\t-detsize         \tdetector size in mm.  may also use -detsize_x -detsize_y\n");
        printf("\t-detpixels       \tdetector size in pixels.  may also use -detpixels_x -detpixels_y\n");
        printf("\t-pixel           \tdetector pixel size in mm.\n");
        printf("\t-Xbeam           \timage X coordinate of direct-beam spot (mm). (default: center)\n");
        printf("\t-Ybeam           \timage Y coordinate of direct-beam spot (mm). )default: center)\n");
        printf("\t-twotheta       \trotation of detector about spindle axis (deg). )default: 0)\n");
        printf("\t-N               \tnumber of unit cells in all directions. may also use -Na -Nb or -Nc\n");
        printf("\t-square_xtal     \tspecify parallelpiped crystal shape (default)\n");
        printf("\t-round_xtal      \tspecify ellipsoidal crystal shape (sort of)\n");
        printf("\t-oversample      \tnumber of sub-pixels per pixel. use this if xtalsize/lambda > distance/pixel\n");
        printf("\t-lambda          \tincident x-ray wavelength in Angstrom. may also use -energy in eV\n");
        printf("\t-dispersion      \tspectral dispersion: delta-lambda/lambda in percent\n");
        printf("\t-dispsteps       \tnumber of wavelengths in above range\n");
        printf("\t-hdivrange       \thorizontal angular spread of source points in mrad\n");
        printf("\t-vdivrange       \tvertical angular spread of source points in mrad\n");
        printf("\t-hdivstep        \tnumber of source points in the horizontal\n");
        printf("\t-vdivstep        \tnumber of source points in the vertical\n");
        printf("\t-square_div      \tfull divergence grid (default: round off corners)\n");
        printf("\t-phi             \tstarting rotation value about spindle axis\n");
        printf("\t-osc             \trotation range about spindle axis\n");
        printf("\t-phisteps        \tnumber of rotation steps to render\n");
        printf("\t-water           \tadd contribution of x microns of water surrounding crystal\n");
        printf("\t-floatfile       \tname of binary output file (4-byte floats)\n");
        printf("\t-intfile         \tname of noiseless smv-formatted output file\n");
        printf("\t-scale           \tscale factor to apply to intfile (default: autoscale)\n");
        printf("\t-noisefile       \tname of photon-scale smv-formatted output file (with Poisson noise)\n");
        printf("\t-roi             \tonly render part of the image: xmin xmax ymin ymax\n");
        printf("\t-printout        \tprint pixel values out to the screen\n");
        printf("\t-seed            \tspecify random-number seed for noisefile\n");
        printf("\t-fluence         \tincident beam intensity for photon-counting statistics (photons/m^2)\n");
        printf("\t-nonoise         \tdisable generating the noisefile\n");
        printf("\t-noprogress      \tturn off the progress meter\n");
        printf("\t-nopolar         \tturn off the polarization correction\n");
        printf("\t-nointerpolate   \tdisable inter-Bragg peak structure factor interpolation\n");
        printf("\t-interpolate     \tforce inter-Bragg peak structure factor interpolation (default: on if < 3 cells wide)\n");
        printf("\t-point_pixel     \tturn off the pixel solid angle correction\n");
        printf("\t-curved_det      \tall pixels same distance from crystal\n");
        printf("\t-binary_spots    \tclip lattice transform at fwhm: no inter-Bragg maxima\n");
        printf("\t-xdet_vector     \tunit vector of increasing x in detector pixel coordinate (default: 0 0 1)\n");
        printf("\t-ydet_vector     \tunit vector of increasing y in detector pixel coordinate (default: 0 -1 0)\n");
        printf("\t-zdet_vector     \tunit vector of increasing detector distance (default: 1 0 0)\n");
        printf("\t-beam_vector     \tunit vector of x-ray beam direction (default: 1 0 0)\n");
        printf("\t-polar_vector    \tunit vector of x-ray E-vector polarization (default: 0 0 1)\n");
        printf("\t-spindle_axis    \tunit vector of right-handed phi rotation axis (default: 0 0 1)\n");
        printf("\t-pix0_vector     \tvector from crystal to first pixel in image (default: beam centered on detector)\n");
        printf("\t-source_distance \tdistance of x-ray source from crystal (default: 10 meters)\n");
	exit(9);
    }

    
    /* allocate memory */
    if(xpixels) {
	detsize_x = pixel*xpixels;
    }
    if(ypixels) {
	detsize_y = pixel*ypixels;
    }
    xpixels = ceil(detsize_x/pixel);
    ypixels = ceil(detsize_y/pixel);
    pixels = xpixels*ypixels;
    floatimage = calloc(pixels+10,sizeof(float));
    //sinimage = calloc(pixels+10,2*sizeof(float));
    //cosimage = calloc(pixels+10,2*sizeof(float));
    intimage = calloc(pixels+10,sizeof(unsigned short int));

    /* defaults? */
    if(Xbeam <= -1e99) Xbeam = (detsize_x - pixel)/2.0;
    if(Ybeam <= -1e99) Ybeam = (detsize_y + pixel)/2.0;
    if(detector_twotheta > -1e99) {
        rotate_axis(xdet_vector,newvector,spindle_vector,detector_twotheta);
        xdet_vector[1] = newvector[1];
        xdet_vector[2] = newvector[2];
        xdet_vector[3] = newvector[3];
        rotate_axis(ydet_vector,newvector,spindle_vector,detector_twotheta);
        ydet_vector[1] = newvector[1];
        ydet_vector[2] = newvector[2];
        ydet_vector[3] = newvector[3];
        rotate_axis(zdet_vector,newvector,spindle_vector,detector_twotheta);
        zdet_vector[1] = newvector[1];
        zdet_vector[2] = newvector[2];
        zdet_vector[3] = newvector[3];
    }
    if(pix0_vector[0] == 0.0) {
        pix0_vector[1] = -Xbeam*xdet_vector[1]-Ybeam*ydet_vector[1]+distance*zdet_vector[1];
        pix0_vector[2] = -Xbeam*xdet_vector[2]-Ybeam*ydet_vector[2]+distance*zdet_vector[2];
        pix0_vector[3] = -Xbeam*xdet_vector[3]-Ybeam*ydet_vector[3]+distance*zdet_vector[3];
    }
    if(detector_twotheta <= -1e99) detector_twotheta = 0;
    if(roi_xmin < 0) roi_xmin = 0;
    if(roi_xmax < 0) roi_xmax = xpixels;
    if(roi_ymin < 0) roi_ymin = 0;
    if(roi_ymax < 0) roi_ymax = ypixels;
    progress_pixels = (roi_xmax-roi_xmin+1)*(roi_ymax-roi_ymin+1);


    if(phisteps < 0){
        /* auto-select number of phi steps */
	if(osc < 0.0) {
            /* auto-select osc range */
            if(phistep <= 0.0) {
	        /* user doesn't care about anything */
	        phisteps = 1;
		osc = 0.0;
		phistep = 0.0;
	    } else {
		/* user doesn't care about osc or steps, but specified step */
		osc = phistep;
		phisteps = 2;
	    }
	} else {
	    /* user-speficied oscillation */
	    if(phistep <= 0.0) {
	        /* osc specified, but nothing else */
                phisteps = 2;
                phistep = osc/2.0;
            } else {
                /* osc and phi step specified */
                phisteps = ceil(osc/phistep);
	    }
	}
    } else {
	/* user-specified number of phi steps */
	if(phisteps == 0) phisteps = 1;
	if(osc < 0.0) {
            /* auto-select osc range */
            if(phistep <= 0.0) {
	        /* user cares only about number of steps */
		osc = 1.0/RTD;
		phistep = osc/phisteps;
	    } else {
		/* user doesn't care about osc, but specified step */
		osc = phistep;
		phisteps = 2;
	    }
	} else {
	    /* user-speficied oscillation */
	    if(phistep < 0.0) {
	        /* osc and steps specified */
                phistep = osc/phisteps;
            } else {
                /* everything specified */
	    }
	}
    }

    if(hdivsteps <= 0){
        /* auto-select number of steps */
	if(hdivrange < 0.0) {
            /* auto-select range */
            if(hdivstep <= 0.0) {
	        /* user doesn't care about anything */
	        hdivsteps = 1;
		hdivrange = 0.0;
		hdivstep = 0.0;
	    } else {
		/* user specified stepsize and nothing else */
		hdivrange = hdivstep;
		hdivsteps = 2;
	    }
	} else {
	    /* user-speficied range */
	    if(hdivstep <= 0.0) {
	        /* range specified, but nothing else */
                hdivstep = hdivrange;
                hdivsteps = 2;
            } else {
                /* range and step specified, but not number of steps */
                hdivsteps = ceil(hdivrange/hdivstep);
	    }
	}
    } else {
	/* user-specified number of steps */
	if(hdivrange < 0.0) {
            /* auto-select range */
            if(hdivstep <= 0.0) {
	        /* user cares only about number of steps */
		hdivrange = 1.0;
		hdivstep = hdivrange/hdivsteps;
	    } else {
		/* user doesn't care about range */
		hdivrange = hdivstep;
		hdivsteps = 2;
	    }
	} else {
	    /* user-speficied range */
	    if(hdivstep <= 0.0) {
	        /* range and steps specified */
		if(hdivsteps <=1 ) hdivsteps = 2;
                hdivstep = hdivrange/(hdivsteps-1);
            } else {
                /* everything specified */
	    }
	}
    }

    if(vdivsteps <= 0){
        /* auto-select number of steps */
	if(vdivrange < 0.0) {
            /* auto-select range */
            if(vdivstep <= 0.0) {
	        /* user doesn't care about anything */
	        vdivsteps = 1;
		vdivrange = 0.0;
		vdivstep = 0.0;
	    } else {
		/* user specified stepsize and nothing else */
		vdivrange = vdivstep;
		vdivsteps = 2;
	    }
	} else {
	    /* user-speficied range */
	    if(vdivstep <= 0.0) {
	        /* range specified, but nothing else */
                vdivstep = vdivrange;
                vdivsteps = 2;
            } else {
                /* range and step specified, but not number of steps */
                vdivsteps = ceil(vdivrange/vdivstep);
	    }
	}
    } else {
	/* user-specified number of steps */
	if(vdivrange < 0.0) {
            /* auto-select range */
            if(vdivstep <= 0.0) {
	        /* user cares only about number of steps */
		vdivrange = 1.0;
		vdivstep = vdivrange/vdivsteps;
	    } else {
		/* user doesn't care about range */
		vdivrange = vdivstep;
		vdivsteps = 2;
	    }
	} else {
	    /* user-speficied range */
	    if(vdivstep <= 0.0) {
	        /* range and steps specified */
		if(vdivsteps <=1 ) vdivsteps = 2;
                vdivstep = vdivrange/(vdivsteps-1);
            } else {
                /* everything specified */
	    }
	}
    }
    

    if(dispsteps <= 0){
        /* auto-select number of steps */
	if(dispersion < 0.0) {
            /* auto-select range */
            if(dispstep <= 0.0) {
	        /* user doesn't care about anything */
	        dispsteps = 1;
		dispersion = 0.0;
		dispstep = 0.0;
	    } else {
		/* user specified stepsize and nothing else */
		dispersion = dispstep;
		dispsteps = 2;
	    }
	} else {
	    /* user-speficied range */
	    if(dispstep <= 0.0) {
	        /* range specified, but nothing else */
                dispstep = dispersion;
                dispsteps = 2;
            } else {
                /* range and step specified, but not number of steps */
                dispsteps = ceil(dispersion/dispstep);
	    }
	}
    } else {
	/* user-specified number of steps */
	if(dispersion < 0.0) {
            /* auto-select range */
            if(dispstep <= 0.0) {
	        /* user cares only about number of steps */
		dispersion = 1.0;
		dispstep = dispersion/dispsteps;
	    } else {
		/* user doesn't care about range */
		dispersion = dispstep;
		dispsteps = 2;
	    }
	} else {
	    /* user-speficied range */
	    if(dispstep <= 0.0) {
	        /* range and steps specified */
		if(dispsteps <=1 ) dispsteps = 2;
                dispstep = dispersion/(dispsteps-1);
            } else {
                /* everything specified */
	    }
	}
    }
    
    /* sanity checks */
    if(hdivrange <= 0.0 || hdivstep <= 0.0 || hdivsteps <= 0) {
        hdivsteps = 1;
        hdivrange = 0.0;
        hdivstep = 0.0;
    }
    if(vdivrange <= 0.0 || vdivstep <= 0.0 || vdivsteps <= 0) {
        vdivsteps = 1;
        vdivrange = 0.0;
        vdivstep = 0.0;
    }
    if(dispersion <= 0.0 || dispstep <= 0.0 || dispsteps <= 0) {
        dispsteps = 1;
        dispersion = 0.0;
        dispstep = 0.0;
    }


    if(interpolate > 1){
	/* no user options */
	if(( Na <= 2) || (Nb <= 2) || (Nc <= 2)){
	    printf("auto-selected tricubic interpolation of structure factors\n");
	    interpolate = 1;
	}
	else
	{
	    printf("auto-selected no interpolation\n");
	    interpolate = 0;
	}
    }

    /* load the lattice orientation (reciprocal cell vectors) */
    printf("reading %s\n",matfilename);
    if(! fscanf(matfile,"%lg%lg%lg",&a_star_x,&b_star_x,&c_star_x)) {perror("fscanf");};
    if(! fscanf(matfile,"%lg%lg%lg",&a_star_y,&b_star_y,&c_star_y)) {perror("fscanf");};
    if(! fscanf(matfile,"%lg%lg%lg",&a_star_z,&b_star_z,&c_star_z)) {perror("fscanf");};
    fclose(matfile);

    /* these include the wavelength */
    a_star_x /=  lambda0*1e10;
    a_star_y /=  lambda0*1e10;
    a_star_z /=  lambda0*1e10;
    b_star_x /=  lambda0*1e10;
    b_star_y /=  lambda0*1e10;
    b_star_z /=  lambda0*1e10;
    c_star_x /=  lambda0*1e10;
    c_star_y /=  lambda0*1e10;
    c_star_z /=  lambda0*1e10;

    /* reciprocal cell lengths */
    a_star = sqrt(a_star_x*a_star_x+a_star_y*a_star_y+a_star_z*a_star_z);
    b_star = sqrt(b_star_x*b_star_x+b_star_y*b_star_y+b_star_z*b_star_z);
    c_star = sqrt(c_star_x*c_star_x+c_star_y*c_star_y+c_star_z*c_star_z);

    /* various cross products */
    a_star_cross_b_star_x = a_star_y*b_star_z - a_star_z*b_star_y;
    a_star_cross_b_star_y = a_star_z*b_star_x - a_star_x*b_star_z;
    a_star_cross_b_star_z = a_star_x*b_star_y - a_star_y*b_star_x;

    b_star_cross_c_star_x = b_star_y*c_star_z - b_star_z*c_star_y;
    b_star_cross_c_star_y = b_star_z*c_star_x - b_star_x*c_star_z;
    b_star_cross_c_star_z = b_star_x*c_star_y - b_star_y*c_star_x;

    c_star_cross_a_star_x = c_star_y*a_star_z - c_star_z*a_star_y;
    c_star_cross_a_star_y = c_star_z*a_star_x - c_star_x*a_star_z;
    c_star_cross_a_star_z = c_star_x*a_star_y - c_star_y*a_star_x;

    /* reciprocal unit cell volume */
    V_star = a_star_x*b_star_cross_c_star_x + a_star_y*b_star_cross_c_star_y + a_star_z*b_star_cross_c_star_z;

    /* direct-space cell volume */
    V_cell = 1.0/V_star;

    /* reciprocal lattice vector "a_star" is defined as perpendicular to both b and c, and must also preserve volume
       converse is true for direct-space lattice: a is perpendicular to both b_star and c_star
       a = ( b_star cross c_star ) / V_star    */

    /* direct-space cell vectors */
    a_x = b_star_cross_c_star_x/V_star;
    a_y = b_star_cross_c_star_y/V_star;
    a_z = b_star_cross_c_star_z/V_star;

    b_x = c_star_cross_a_star_x/V_star;
    b_y = c_star_cross_a_star_y/V_star;
    b_z = c_star_cross_a_star_z/V_star;

    c_x = a_star_cross_b_star_x/V_star;
    c_y = a_star_cross_b_star_y/V_star;
    c_z = a_star_cross_b_star_z/V_star;

    /* calculate direct-space unit cell lengths */
    a = sqrt(a_x*a_x+a_y*a_y+a_z*a_z);
    b = sqrt(b_x*b_x+b_y*b_y+b_z*b_z);
    c = sqrt(c_x*c_x+c_y*c_y+c_z*c_z);

    /* for fun, calculate the cell angles too */
    sinalpha = a_star*V_cell/b/c;
    sinbeta  = b_star*V_cell/a/c;
    singamma = c_star*V_cell/a/b;
    if(sinalpha*sinalpha>1.0) sinalpha=1.0;
    if(sinbeta*sinbeta>1.0)   sinbeta=1.0;
    if(singamma*singamma>1.0) singamma=1.0;
    alpha = RTD*atan2(sinalpha,sqrt(1.0-sinalpha*sinalpha));
    beta  = RTD*atan2(sinbeta ,sqrt(1.0-sinbeta*sinbeta));
    gamma = RTD*atan2(singamma,sqrt(1.0-singamma*singamma));

    /* reciprocal cell angles */
    sinalpha_star = a*V_star/b_star/c_star;
    sinbeta_star  = b*V_star/a_star/c_star;
    singamma_star = c*V_star/a_star/b_star;
    if(sinalpha_star*sinalpha_star>1.0) sinalpha_star=1.0;
    if(sinbeta_star*sinbeta_star>1.0)   sinbeta_star=1.0;
    if(singamma_star*singamma_star>1.0) singamma_star=1.0;
    alpha_star = RTD*atan2(sinalpha_star,sqrt(1.0-sinalpha_star*sinalpha_star));
    beta_star  = RTD*atan2(sinbeta_star ,sqrt(1.0-sinbeta_star*sinbeta_star));
    gamma_star = RTD*atan2(singamma_star,sqrt(1.0-singamma_star*singamma_star));

    printf("Unit Cell: %g %g %g %g %g %g\n", a,b,c,alpha,beta,gamma);
    printf("volume = %g A^3\n",V_cell);

    /* print out the real-space matrix */
    printf("real-space cell vectors (Angstrom):\n");
    printf("     %-10s  %-10s  %-10s\n","a","b","c");
    printf("X: %11.8f %11.8f %11.8f\n",a_x,b_x,c_x);
    printf("Y: %11.8f %11.8f %11.8f\n",a_y,b_y,c_y);
    printf("Z: %11.8f %11.8f %11.8f\n",a_z,b_z,c_z);

    /* now convert these to meters */
    a*=1e-10;b*=1e-10;c*=1e-10;
    a_x*=1e-10;b_x*=1e-10;c_x*=1e-10;
    a_y*=1e-10;b_y*=1e-10;c_y*=1e-10;
    a_z*=1e-10;b_z*=1e-10;c_z*=1e-10;

    /* define phi=0 crystal orientation */
    a_x0=a_x;a_y0=a_y;a_z0=a_z;
    b_x0=b_x;b_y0=b_y;b_z0=b_z;
    c_x0=c_x;c_y0=c_y;c_z0=c_z;

    /* calculate crystal size in meters */
    xtalsize_a = a*Na;
    xtalsize_b = b*Nb;
    xtalsize_c = c*Nc;
    printf("crystal is %g x %g x %g microns\n",xtalsize_a*1e6,xtalsize_b*1e6,xtalsize_c*1e6);
    xtalsize_max = xtalsize_a;
    if(xtalsize_max < xtalsize_b) xtalsize_max = xtalsize_b;
    if(xtalsize_max < xtalsize_c) xtalsize_max = xtalsize_c;
    if(oversample <= 0) {
	oversample = ceil(3.0 * xtalsize_max/lambda0 / (distance/pixel));
	if(oversample <= 0) oversample = 1;
	printf("auto-selected %d-fold oversampling\n",oversample);
    }


    /* load the structure factors */
    if(hklfile == NULL)
    {
	/* try to recover Fs from a previous run */
	dumpfile = fopen(dumpfilename,"r");
 	if(dumpfile == NULL)
	{
	    printf("ERROR: no hkl file and no dump file to read.");
	    exit(9);
	}
	printf("reading Fs from %s\n",dumpfilename);
	n=0;
	if(! fscanf(dumpfile,"%d%d%d%d%d%d\n\f",&h_min,&h_max,&k_min,&k_max,&l_min,&l_max) ) {perror("fscanf");};
        h_range = h_max - h_min + 1;
        k_range = k_max - k_min + 1;
        l_range = l_max - l_min + 1;
	Fhkl = (double***) calloc(h_range+1,sizeof(double**));
	for (h0=0; h0<=h_range;h0++) {
	    *(Fhkl +h0) = (double**) calloc(k_range+1,sizeof(double*));
	    for (k0=0; k0<=k_range;k0++) {
		*(*(Fhkl +h0)+k0) = (double*) calloc(l_range+1,sizeof(double));
		if(! fread(*(*(Fhkl +h0)+k0),sizeof(double),l_range+1,dumpfile) )
		{
		    perror("fscanf");
		};
		n+=l_range;
	    }
	}
	fclose(dumpfile);
    }
    else
    {
    	n = 0;
        h_min=k_min=l_min=1e9;
        h_max=k_max=l_max=-1e9;
        printf("counting entries in %s\n",hklfilename);
        while(4 == fscanf(hklfile,"%lg%lg%lg%lg",&h,&k,&l,&F_cell)){
	    if(h != ceil(h-0.4)) printf("WARNING: non-integer value for h (%g) at line %d\n",h,n);
	    if(k != ceil(k-0.4)) printf("WARNING: non-integer value for k (%g) at line %d\n",k,n);
	    if(l != ceil(l-0.4)) printf("WARNING: non-integer value for l (%g) at line %d\n",l,n);
	    if(h_min > h) h_min = h;
	    if(k_min > k) k_min = k;
	    if(l_min > l) l_min = l;
	    if(h_max < h) h_max = h;
	    if(k_max < k) k_max = k;
	    if(l_max < l) l_max = l;
	    ++n;
        }
        rewind(hklfile);
        h_range = h_max - h_min + 1;
        k_range = k_max - k_min + 1;
        l_range = l_max - l_min + 1;

	if(h_range < 0 || k_range < 0 || l_range < 0) {
	    printf("h: %d - %d\n",h_min,h_max);
	    printf("k: %d - %d\n",k_min,k_max);
	    printf("l: %d - %d\n",l_min,l_max);
	    printf("ERROR: not enough HKL indices in %s\n",hklfilename);
	    exit(9);
 	}

	/* allocate memory for 3d arrays */
	//printf("allocating %d %d-byte double**\n",h_range+1,sizeof(double**));
	Fhkl = (double***) calloc(h_range+1,sizeof(double**));
	if(Fhkl==NULL){perror("ERROR");exit(9);};
	for (h0=0; h0<=h_range;h0++) {
		//printf("allocating %d %d-byte double*\n",k_range+1,sizeof(double*));
		Fhkl[h0] = (double**) calloc(k_range+1,sizeof(double*));
		if(Fhkl[h0]==NULL){perror("ERROR");exit(9);};
		for (k0=0; k0<=k_range;k0++) {
			//printf("allocating %d %d-byte double\n",k_range+1,sizeof(double));
			Fhkl[h0][k0] = (double*) calloc(l_range+1,sizeof(double));
			if(Fhkl[h0][k0]==NULL){perror("ERROR");exit(9);};
		}
	}


        printf("re-reading %s\n",hklfilename);
        while(4 == fscanf(hklfile,"%d%d%d%lg",&h0,&k0,&l0,&F_cell)){
	    Fhkl[h0-h_min][k0-k_min][l0-l_min]=F_cell;
    	}
    	fclose(hklfile);

//	for(h0=h_min;h0<=h_max;++h0){
//	    for(k0=k_min;k0<=k_max;++k0){
//		for(l0=l_min;l0<=l_max;++l0){
//		    if ( (h0<=h_max) && (h0>=h_min) && (k0<=k_max) && (k0>=k_min) && (l0<=l_max) && (l0>=l_min)  ) {
//		    	/* just take nearest-neighbor */
//		    	F_cell = Fhkl[h0-h_min][k0-k_min][l0-l_min];
//		    }
//		    else
//		    {
//		    	F_cell = 0.0;
//		    }
//		    printf("%d %d %d = %f\n",h0,k0,l0,F_cell);
//		}
//	    }
//	}

	/* make dump file */
	dumpfile = fopen(dumpfilename,"w");
	if(dumpfile == NULL)
	{
	    printf("WARNING: unable to open dump file: %s\n",dumpfilename);
	}
	else
	{
	    printf("writing dump file for next time: %s\n",dumpfilename);
	    fprintf(dumpfile,"%d %d %d %d %d %d\n\f",h_min,h_max,k_min,k_max,l_min,l_max);
	    for (h0=0; h0<=h_range;h0++) {
		for (k0=0; k0<=k_range;k0++) {
			fwrite(*(*(Fhkl +h0)+k0),sizeof(double),l_range+1,dumpfile);
		}
	    }
	    fclose(dumpfile);
	}
    }

    if(interpolate){
        /* allocate interpolation array */
	sub_Fhkl = (double***) calloc(6,sizeof(double**));
	for (h0=0; h0<=5;h0++) {
		*(sub_Fhkl +h0) = (double**) calloc(6,sizeof(double*));
		for (k0=0; k0<=5;k0++) {
			*(*(sub_Fhkl +h0)+k0) = (double*) calloc(6,sizeof(double));
		}
	}
    }


    /* count divsteps sweep over solid angle of beam divergence */
    divsteps = 0;
    for(hdiv_tic=0;hdiv_tic<hdivsteps;++hdiv_tic){
        for(vdiv_tic=0;vdiv_tic<vdivsteps;++vdiv_tic){                
	    hdiv = hdivstep * hdiv_tic - hdivrange/2.0 ;
	    vdiv = vdivstep * vdiv_tic - vdivrange/2.0 ;
	    /* force an elliptical divergence */
	    test = (hdiv*hdiv-hdivstep*hdivstep/4.0*(1-hdivsteps%2))/hdivrange/hdivrange ;
	    test += (vdiv*vdiv-vdivstep*vdivstep/4.0*(1-vdivsteps%2))/vdivrange/vdivrange ;
	    if( round_div && test*4.0 > 1.1) continue;
                
	    ++divsteps;
            printf("divergence deviation: %g %g\n",hdiv,vdiv);
        }
    }

    /* print out wavelength steps with sweep over spectral dispersion */
    for(disp_tic=0;disp_tic<dispsteps;++disp_tic){
	lambda = lambda0 * ( 1.0 + dispstep * disp_tic - dispersion/2.0 ) ;
	printf("lambda%d = %.15g\n",disp_tic,lambda);
    }

    /* count phi steps with sweep over spindle axis */
    for(phi_tic = 0; phi_tic < phisteps; ++phi_tic){
	phi = phi0 + phistep*phi_tic;
	printf("phi%d = %g\n",phi_tic,phi*RTD);
    }

    /* total number of steps */
    steps = phisteps*divsteps*dispsteps*oversample*oversample;
    subpixel = pixel/oversample;

    printf("  %d initialized hkls (all others zero)\n",n);
    printf("  ");
    if(round_xtal){
	printf("ellipsoidal");
    }
    else
    {
	printf("parallelpiped");
    }
    printf(" xtal: %.0fx%.0fx%.0f cells\n",Na,Nb,Nc);
    printf("  wave=%g meters +/- %g%% in %d steps\n",lambda0,dispersion*100,dispsteps);
    if(nopolar) printf("polarization effect disabled\n");
    if(curved_detector) printf("curved detector: all pixels same distance from origin\n");
    if(point_pixel) printf("pixel obliquity effect disabled\n");
    printf("  incident fluence: %g photons/m^2\n",fluence);
    printf("  distance=%g detsize=%gx%g  pixel=%g meters (%dx%d pixels)\n",distance,detsize_x,detsize_y,pixel,xpixels,ypixels);
    printf("  Xbeam=%g Ybeam=%g\n",Xbeam,Ybeam);
    printf("  roi: %d < x < %d && %d < y < %d\n",roi_xmin,roi_xmax,roi_ymin,roi_ymax);
    printf("  hdivrange=%g hdivstep=%g  radians\n",hdivrange,hdivstep);
    printf("  vdivrange=%g vdivstep=%g  radians\n",vdivrange,vdivstep);
    printf("  %d divergence steps\n",divsteps);
    printf("  %d phi steps from %g to %g degrees\n",phisteps,phi0*RTD,(phi0+osc)*RTD);
    printf("  %dx%d pixel oversample steps\n",oversample,oversample);
//    printf("  coherent source: %d\n",coherent);
    if(calculate_noise){
        printf("\n  noise image paramters:\n");
        printf("  seed: %ld\n",seed);
        printf("  water droplet size: %g m\n",water_size);
    } 


    /* initialize the incident beam unit vector */
    S0_x = 1;
    S0_y = 0;
    S0_z = 0;

    /* sweep over detector */   
    j = 0;
    progress_pixel = 0;
    for(ypixel=0;ypixel<ypixels;++ypixel){
     for(xpixel=0;xpixel<xpixels;++xpixel){

      if(xpixel < roi_xmin || xpixel > roi_xmax || ypixel < roi_ymin || ypixel > roi_ymax) {
	   ++j; continue;
      }

      /* reset photon count */
      I = 0;

      for(suby=0;suby<oversample;++suby){
       for(subz=0;subz<oversample;++subz){

	Xdet = subpixel*(xpixel*oversample + subz ) + subpixel/2.0;
	Ydet = subpixel*(ypixel*oversample + suby ) + subpixel/2.0;
//      Xdet = pixel*xpixel;
//	Ydet = pixel*ypixel;

	/* construct detector pixel position */
//        pixel_X = distance;
//        pixel_Y = Ydet-Ybeam;
//        pixel_Z = Xdet-Xbeam;
        pixel_X = Xdet*xdet_vector[1]+Ydet*ydet_vector[1]+pix0_vector[1];
        pixel_Y = Xdet*xdet_vector[2]+Ydet*ydet_vector[2]+pix0_vector[2];
        pixel_Z = Xdet*xdet_vector[3]+Ydet*ydet_vector[3]+pix0_vector[3];
	if(curved_detector) {
	    /* construct detector pixel that is always "distance" from the origin */
	    vector[1] = distance; vector[2]=0 ; vector[3]=0;
            rotate(vector,newvector,0,pixel_Z/distance,pixel_Y/distance);
            pixel_X = newvector[1];
            pixel_Y = newvector[2];
            pixel_Z = newvector[3];
	}
	/* construct the unit vector to this pixel */
	airpath = sqrt(pixel_X*pixel_X+pixel_Y*pixel_Y+pixel_Z*pixel_Z);
	S_x = pixel_X/airpath;
	S_y = pixel_Y/airpath;
	S_z = pixel_Z/airpath;

	/* rough cut to speed things up when we aren't using whole detector */
	if(dmin>0){
	    s = sqrt((S_x-1)*(S_x-1)+S_y*S_y+S_z*S_z)/lambda0;
	    if(s>0.0){
		if(dmin > 1.0/s){
		    continue;
		}
	    }
	}

	/* solid angle subtended by a pixel: (pix/airpath)^2*cos(2theta) */
	omega_pixel = pixel*pixel/airpath/airpath*distance/airpath;
	if(point_pixel) omega_pixel = 1.0/airpath/airpath;

	/* polarization factor for this pixel */
	if(! nopolar){
	    costwotheta = pixel_X/airpath;
	    polar = 0.5*(1.0+costwotheta*costwotheta);
	}else{
	    polar = 1.0;
	}

	/* sweep over wavelengths */
        for(disp_tic=0;disp_tic<dispsteps;++disp_tic){
           lambda = lambda0 * ( 1.0 + dispstep * disp_tic - dispersion/2.0 ) ;

           /* sweep over solid angle of beam divergence */
           for(hdiv_tic=0;hdiv_tic<hdivsteps;++hdiv_tic){
             for(vdiv_tic=0;vdiv_tic<vdivsteps;++vdiv_tic){                
	        hdiv = hdivstep * hdiv_tic - hdivrange/2.0 ;
	        vdiv = vdivstep * vdiv_tic - vdivrange/2.0 ;
	        /* force an elliptical divergence */
	        test = (hdiv*hdiv-hdivstep*hdivstep/4.0*(1-hdivsteps%2))/hdivrange/hdivrange ;
	        test += (vdiv*vdiv-vdivstep*vdivstep/4.0*(1-vdivsteps%2))/vdivrange/vdivrange ;
	        if( round_div && test*4.0 > 1.1) continue;

		/* construct source position (flat, coherently-emitting plane) */
		source_X = -source_distance;
		source_Y = atan(hdiv)*source_distance;
		source_Z = atan(vdiv)*source_distance;

		/* construct the incident beam unit vector */
		source_path = sqrt(source_X*source_X+source_Y*source_Y+source_Z*source_Z);
		S0_x = -source_X/source_path;
		S0_y = -source_Y/source_path;
		S0_z = -source_Z/source_path;
		

		/* construct the scattering vector for this pixel */
		s_x = (S_x-S0_x)/lambda;
		s_y = (S_y-S0_y)/lambda;
		s_z = (S_z-S0_z)/lambda;

        	/* sweep over phi angles */
                for(phi_tic = 0; phi_tic < phisteps; ++phi_tic){
		    phi = phi0 + phistep*phi_tic;
		    
		    if( phi != 0.0 ) {
		        /* rotate about spindle */
		        vector[1]=a_x0; vector[2]=a_y0; vector[3]=a_z0;
        	        rotate_axis(vector,newvector,spindle_vector,phi);
        	        a_x = newvector[1];
        	        a_y = newvector[2];
        	        a_z = newvector[3];
		        vector[1]=b_x0; vector[2]=b_y0; vector[3]=b_z0;
        	        rotate_axis(vector,newvector,spindle_vector,phi);
        	        b_x = newvector[1];
        	        b_y = newvector[2];
          	        b_z = newvector[3];
	  	        vector[1]=c_x0; vector[2]=c_y0; vector[3]=c_z0;
          	        rotate_axis(vector,newvector,spindle_vector,phi);
          	        c_x = newvector[1];
          	        c_y = newvector[2];
          	        c_z = newvector[3];
          	    }

		    /* construct fractional Miller indicies */
		    h = a_x*s_x + a_y*s_y + a_z*s_z;
		    k = b_x*s_x + b_y*s_y + b_z*s_z;
		    l = c_x*s_x + c_y*s_y + c_z*s_z;

		    /* round off to nearest whole index */
		    h0 = ceil(h-0.5);
		    k0 = ceil(k-0.5);
		    l0 = ceil(l-0.5);

		    /* find nearest point on Ewald sphere surface? */
		    if( integral_form ) {

		      if( phi != 0.0 ) {
			/* need to re-calculate reciprocal matrix */

          	        /* various cross products */
           	        a_cross_b_x = a_y*b_z - a_z*b_y;
          	        a_cross_b_y = a_z*b_x - a_x*b_z;
          	        a_cross_b_z = a_x*b_y - a_y*b_x;

          	        b_cross_c_x = b_y*c_z - b_z*c_y;
           	        b_cross_c_y = b_z*c_x - b_x*c_z;
           	        b_cross_c_z = b_x*c_y - b_y*c_x;

           	        c_cross_a_x = c_y*a_z - c_z*a_y;
           	        c_cross_a_y = c_z*a_x - c_x*a_z;
           	        c_cross_a_z = c_x*a_y - c_y*a_x;

                        /* reciprocal-space cell vectors */
           	        a_star_x = b_cross_c_x/V_cell*1e20;
                        a_star_y = b_cross_c_y/V_cell*1e20;
                        a_star_z = b_cross_c_z/V_cell*1e20;

                        b_star_x = c_cross_a_x/V_cell*1e20;
                        b_star_y = c_cross_a_y/V_cell*1e20;
                        b_star_z = c_cross_a_z/V_cell*1e20;

                        c_star_x = a_cross_b_x/V_cell*1e20;
                        c_star_y = a_cross_b_y/V_cell*1e20;
                        c_star_z = a_cross_b_z/V_cell*1e20;
		      }
	  
		      /* reciprocal-space coordinates of relp */
                      X = h0*a_star_x + k0*b_star_x + l0*c_star_x;
                      Y = h0*a_star_y + k0*b_star_y + l0*c_star_y;
                      Z = h0*a_star_z + k0*b_star_z + l0*c_star_z;
//		      d_star = sqrt(X^2 + Y^2 + Z^2)

		      /* reciprocal-space coordinates of center of Ewald sphere */
                      X0 = -S0_x/lambda/1e10;
                      Y0 = -S0_y/lambda/1e10;
                      Z0 = -S0_z/lambda/1e10;
//		      1 = sqrt(X0^2 + Y0^2 + Z0^2)

		      /* distance from Ewald sphere in lambda=1 units */
		      d_r = sqrt((X-X0)*(X-X0)+(Y-Y0)*(Y-Y0)+(Z-Z0)*(Z-Z0))-1.0;

		      /* unit vector of diffracted ray through relp */
		      diffray_x = (X - X0)/(1+d_r);
		      diffray_y = (Y - Y0)/(1+d_r);
		      diffray_z = (Z - Z0)/(1+d_r);

		      /* intersection with detector plane */
		      xd = xdet_vector[1]*diffray_x + xdet_vector[2]*diffray_y + xdet_vector[3]*diffray_z;
		      yd = ydet_vector[1]*diffray_x + ydet_vector[2]*diffray_y + ydet_vector[3]*diffray_z;
		      zd = zdet_vector[1]*diffray_x + zdet_vector[2]*diffray_y + zdet_vector[3]*diffray_z;

		      /* where does the central direct-beam hit */
		      xd0 = xdet_vector[1]*S0_x + xdet_vector[2]*S0_y + xdet_vector[3]*S0_z;
		      yd0 = ydet_vector[1]*S0_x + ydet_vector[2]*S0_y + ydet_vector[3]*S0_z;
		      zd0 = zdet_vector[1]*S0_x + zdet_vector[2]*S0_y + zdet_vector[3]*S0_z;

		      /* convert to mm coordinates */
		      Xdet0 = distance*(xd/zd) + Xbeam;
		      Ydet0 = distance*(yd/zd) + Ybeam;
	    
			//printf("GOTHERE %g %g   %g %g\n",Xdet,Ydet,Xdet0,Ydet0);
			test = exp(-( (Xdet-Xdet0)*(Xdet-Xdet0)+(Ydet-Ydet0)*(Ydet-Ydet0) + d_r*d_r )/1e-8);
                    }

		    /* structure factor of the unit cell */
		    if(interpolate){
		        h0_flr = floor(h);
		        k0_flr = floor(k);
		        l0_flr = floor(l);


		        if ( ((h-h_min+3)>h_range) || (h-2<h_min) || ((k-k_min+3)>k_range) || (k-2<k_min) || ((l-l_min+3)>l_range) || (l-2<l_min)  ) {
			    if(babble){
			        babble=0;
			        printf ("WARNING: out of range for three point interpolation: h,k,l,h0,k0,l0: %g,%g,%g,%d,%d,%d \n", h,k,l,h0,k0,l0);
			        printf("WARNING: further warnings will not be printed! ");
			    }
			    continue;
		        }

		        /* integer versions of nearest HKL indicies */
		        h_interp[0]=h0_flr-1;
		        h_interp[1]=h0_flr;
		        h_interp[2]=h0_flr+1;
		        h_interp[3]=h0_flr+2;
		        k_interp[0]=k0_flr-1;
		        k_interp[1]=k0_flr;
		        k_interp[2]=k0_flr+1;
		        k_interp[3]=k0_flr+2;
		        l_interp[0]=l0_flr-1;
		        l_interp[1]=l0_flr;
		        l_interp[2]=l0_flr+1;
		        l_interp[3]=l0_flr+2;
	
		        /* polin function needs doubles */
		        h_interp_d[0] = (double) h_interp[0];
		        h_interp_d[1] = (double) h_interp[1];
		        h_interp_d[2] = (double) h_interp[2];
		        h_interp_d[3] = (double) h_interp[3];
		        k_interp_d[0] = (double) k_interp[0];
		        k_interp_d[1] = (double) k_interp[1];
		        k_interp_d[2] = (double) k_interp[2];
		        k_interp_d[3] = (double) k_interp[3];
		        l_interp_d[0] = (double) l_interp[0];
		        l_interp_d[1] = (double) l_interp[1];
		        l_interp_d[2] = (double) l_interp[2];
		        l_interp_d[3] = (double) l_interp[3];
		
		        /* now populate the "y" values (nearest four structure factors in each direction) */
		        for (i1=0;i1<4;i1++) {
			    for (i2=0;i2<4;i2++) {
		               for (i3=0;i3<4;i3++) {
			  	    sub_Fhkl[i1][i2][i3]=Fhkl[h_interp[i1]-h_min][k_interp[i2]-k_min][l_interp[i3]-l_min];
		               }
		            }
		         }		

		        /* run the tricubic polynomial interpolation */
		        polin3(h_interp_d,k_interp_d,l_interp_d,sub_Fhkl,h,k,l,&F_cell);
		    }
		    else
		    {
		        if ( (h0<=h_max) && (h0>=h_min) && (k0<=k_max) && (k0>=k_min) && (l0<=l_max) && (l0>=l_min)  ) {
		    	    /* just take nearest-neighbor */
		    	    F_cell = Fhkl[h0-h_min][k0-k_min][l0-l_min];
		        }
		        else
		        {
		    	    F_cell = 0.0;
		        }
		    }


		    /* structure factor of the lattice (paralelpiped crystal)
		       F_latt = sin(PI*Na*h)*sin(PI*Nb*k)*sin(PI*Nc*l)/sin(PI*h)/sin(PI*k)/sin(PI*l);
		*/
		    F_latt = 1.0;
		    if(round_xtal){
		        /* use sinc3 for elliptical xtal shape */
		        F_latt = Na*Nb*Nc*sinc3(PI*sqrt(Na*Na*(h-h0)*(h-h0) + Nb*Nb*(k-k0)*(k-k0) + Nc*Nc*(l-l0)*(l-l0) ) );
		    }
		    else
		    {
		        /* xtal is a paralelpiped */
                        if(Na>1){
			    F_latt *= sincg(PI*h,Na);
		        }
                        if(Nb>1){
			    F_latt *= sincg(PI*k,Nb);
		        }
                        if(Nc>1){
			    F_latt *= sincg(PI*l,Nc);
		        }
		    }
		    if(binary_spots) {
//if(l>0.5 && l<1.5) printf("%f %f %f = %d %d %d   %f  at %f %f\n",h,k,l,h0,k0,l0,F_latt,pixel_Y,pixel_Z);
		        F_latt = (fabs(F_latt) > 0.5*Na*Nb*Nc)*Na*Nb*Nc;
		    }

		    /* convert amplitudes into intensity (photons per steradian) */
		    I += F_cell*F_cell*F_latt*F_latt;
	        }
             }
            }
          }
         } // oversample y
        } // oversample x
	/* add background from something? */
	I_water = water_scale*fluence*polar*water_size*water_size*water_size*omega_pixel;


	floatimage[j]= r_e_sqr*fluence*polar*I/steps*omega_pixel + I_water;
//      floatimage[j] = test;
	if(floatimage[j] > max_I) {
	    max_I = floatimage[j];
	    max_I_x = Xdet;
	    max_I_y = Ydet;
	}
	if( printout ) {
	    if(xpixel==printout_xpixel && ypixel==printout_ypixel || printout_xpixel < 0)
	    {
		twotheta = atan2(sqrt(pixel_Y*pixel_Y+pixel_Z*pixel_Z),pixel_X);
		stol = sin(twotheta/2.0)/(lambda0*1e10);
	    	printf("%4d %4d : stol = %g\n", xpixel,ypixel,stol);
	    	printf("at %g %g %g\n", pixel_X,pixel_Y,pixel_Z);
	    	printf("hkl= %f %f %f  hkl0= %d %d %d\n", h,k,l,h0,k0,l0);
	    	printf(" F_cell=%g  F_latt=%g   I = %g\n", F_cell,F_latt,I);
	    	printf("I/steps %15.10g\n", I/steps);
	    	printf("polar   %15.10g\n", polar);
	    	printf("omega   %15.10g\n", omega_pixel);
	    	printf("pixel   %15.10g\n", floatimage[j]);
	    }
	}else{
	    if(progress_meter && progress_pixels/100 > 0){
	        if(progress_pixel % ( progress_pixels/20 ) == 0 || ((10*progress_pixel<progress_pixels || 10*progress_pixel>9*progress_pixels) && (progress_pixel % (progress_pixels/100) == 0))){
		    printf("%ld%% done\n",progress_pixel*100/progress_pixels);
	        }
	    }
//	    printf("%d %d\n",j,j % ( pixels/100 ));
//	    if(j % ( pixels/100 ) == 0) printf("%d%% done\n",j*100/pixels);
	}
//	printf("%d ",intimage[j]);
//	printf("%.10g ",floatimage[j]);
//	printf("%.10g ",max_I);
//	printf("\n");
//;fflush(stdout);
	++j;
	++progress_pixel;
     } // ypixel
    } // xpixel
    printf("\n");

    printf("writing %s as %d %lu-byte floats\n",floatfilename,pixels,sizeof(float));
    outfile = fopen(floatfilename,"w");
    if(outfile == NULL)
    {
	    perror("ERROR: fopen");
	    exit(9);
    }
    fwrite(floatimage,sizeof(float),pixels,outfile);
    fclose(outfile);

    /* output as ints */   
    j = 0;
    printf("max_I = %g  at %g %g\n",max_I,max_I_x,max_I_y);
    if(intfile_scale <= 0.0){
	if(max_I > 0.0) intfile_scale = 55000.0/max_I;
    }
    printf("intfile_scale = %g\n",intfile_scale);
    for(ypixel=0;ypixel<ypixels;++ypixel){
      for(xpixel=0;xpixel<xpixels;++xpixel){
        if(xpixel < roi_xmin || xpixel > roi_xmax || ypixel < roi_ymin || ypixel > roi_ymax) {
	   ++j; continue;
        }
	test = floatimage[j] *intfile_scale+40.0;
	if(test > 65535.0) test = 65535.0;
	intimage[j] = (unsigned short int) ( test );
//	printf("%d %d = %d\n",xpixel,ypixel,intimage[j]);
	++j;
      }
    }

    printf("writing %s as %lu-byte integers\n",intfilename,sizeof(unsigned short int));
    outfile = fopen(intfilename,"w");
    if(outfile == NULL)
    {
	    perror("ERROR: fopen");
	    exit(9);
    }
    fprintf(outfile,"{\nHEADER_BYTES=512;\nDIM=2;\nBYTE_ORDER=little_endian;\nTYPE=unsigned_short;\n");
    fprintf(outfile,"SIZE1=%d;\nSIZE2=%d;\nPIXEL_SIZE=%g;\nDISTANCE=%g;\n",xpixels,ypixels,pixel*1000.0,distance*1000.0);
    fprintf(outfile,"WAVELENGTH=%g;\nBEAM_CENTER_X=%g;\nBEAM_CENTER_Y=%g;\n",lambda0*1e10,Xbeam*1000.0,(detsize_y-Ybeam)*1000);
    fprintf(outfile,"PHI=%g;\nOSC_START=%g;\nOSC_RANGE=%g;\n",phi0*RTD,phi0*RTD,osc*RTD);
    fprintf(outfile,"TWOTHETA=%g;\n",-detector_twotheta*RTD);
    fprintf(outfile,"DETECTOR_SN=000;\n");
    fprintf(outfile,"BEAMLINE=fake;\n");
    fprintf(outfile,"}\f");
    while ( ftell(outfile) < 512 ){ fprintf(outfile," "); };
    fwrite(intimage,sizeof(unsigned short int),pixels,outfile);
    fclose(outfile);

    if(calculate_noise == 0){
	return;
    }

    /* simulate Poisson noise */
    j = 0;
    sum = 0.0;
    overloads = 0;
    for(ypixel=0;ypixel<ypixels;++ypixel){
      for(xpixel=0;xpixel<xpixels;++xpixel){
        if(xpixel < roi_xmin || xpixel > roi_xmax || ypixel < roi_ymin || ypixel > roi_ymax) {
	   ++j; continue;
        }
	test = poidev( floatimage[j], &seed );
	sum += test;
	test += 40;
	if(test > 65535.0) {
	    test = 65535.0;
	    ++overloads;
	}
	intimage[j] = (unsigned short int) test;
//	printf("%d %d = %d\n",xpixel,ypixel,intimage[j]);
	++j;
      }
    }
    printf("%.0f photons on noise image (%d overloads)\n",sum,overloads);

    printf("writing %s as %lu-byte integers\n",noisefilename,sizeof(unsigned short int));
    outfile = fopen(noisefilename,"w");
    if(outfile == NULL)
    {
	    perror("ERROR: fopen");
	    exit(9);
    }
    fprintf(outfile,"{\nHEADER_BYTES=512;\nDIM=2;\nBYTE_ORDER=little_endian;\nTYPE=unsigned_short;\n");
    fprintf(outfile,"SIZE1=%d;\nSIZE2=%d;\nPIXEL_SIZE=%g;\nDISTANCE=%g;\n",xpixels,ypixels,pixel*1000.0,distance*1000.0);
    fprintf(outfile,"WAVELENGTH=%g;\nBEAM_CENTER_X=%g;\nBEAM_CENTER_Y=%g;\n",lambda0*1e10,Xbeam*1000.0,(detsize_y-Ybeam)*1000);
    fprintf(outfile,"PHI=%g;\nOSC_START=%g;\nOSC_RANGE=%g;\n",phi0*RTD,phi0*RTD,osc*RTD);
    fprintf(outfile,"TWOTHETA=%g;\n",-detector_twotheta*RTD);
    fprintf(outfile,"DETECTOR_SN=000;\n");
    fprintf(outfile,"BEAMLINE=fake;\n");
    fprintf(outfile,"}\f");
    while ( ftell(outfile) < 512 ){ fprintf(outfile," "); };
    fwrite(intimage,sizeof(unsigned short int),pixels,outfile);
    fclose(outfile);

    return;
}

/* Fourier transform of a grating */
double sincg(double x,double N) {
    if(x==0.0) return N;

    return sin(x*N)/sin(x);
}

/* Fourier transform of a sphere */
double sinc3(double x) {
    if(x==0.0) return 1.0;

    return 3.0*(sin(x)/x-cos(x))/(x*x);
}

double sinc_conv_sinc3(double x) {
    if(x==0.0) return 1.0;

    return 3.0*(sin(x)-x*cos(x))/(x*x*x);
}


double *rotate(double *v, double *new, double phix, double phiy, double phiz) {
    
    double rxx,rxy,rxz,ryx,ryy,ryz,rzx,rzy,rzz;
    double new_x,new_y,new_z,rotated_x,rotated_y,rotated_z;
    
    new_x=v[1];
    new_y=v[2];
    new_z=v[3];
    
    if(phix != 0){
        /* rotate around x axis */
        //rxx= 1;         rxy= 0;         rxz= 0;
        ryx= 0;         ryy= cos(phix); ryz=-sin(phix);
        rzx= 0;         rzy= sin(phix); rzz= cos(phix);
        
        rotated_x = new_x;
        rotated_y = new_y*ryy + new_z*ryz;
        rotated_z = new_y*rzy + new_z*rzz;
        new_x = rotated_x; new_y = rotated_y; new_z = rotated_z;
    }
    
    if(phiy != 0) {
        /* rotate around y axis */
        rxx= cos(phiy); rxy= 0;         rxz= sin(phiy);
        //ryx= 0;         ryy= 1;         ryz= 0;
        rzx=-sin(phiy); rzy= 0;         rzz= cos(phiy);
        
        rotated_x = new_x*rxx + new_y*rxy + new_z*rxz;
        rotated_y = new_y;
        rotated_z = new_x*rzx + new_y*rzy + new_z*rzz;
        new_x = rotated_x; new_y = rotated_y; new_z = rotated_z;
    }
    
    if(phiz != 0){
        /* rotate around z axis */
        rxx= cos(phiz); rxy=-sin(phiz); rxz= 0;
        ryx= sin(phiz); ryy= cos(phiz); ryz= 0;
        //rzx= 0;         rzy= 0;         rzz= 1;
        
        rotated_x = new_x*rxx + new_y*rxy ;
        rotated_y = new_x*ryx + new_y*ryy;
        rotated_z = new_z;
        new_x = rotated_x; new_y = rotated_y; new_z = rotated_z;
    }
    
    new[1]=new_x;
    new[2]=new_y;
    new[3]=new_z;
    
    return new;
}



/* rotate a point about a unit vector axis */
double *rotate_axis(double *v, double *new, double *axis, double phi) {

    double sinphi = sin(phi);
    double cosphi = cos(phi);
    double dot = (axis[1]*v[1]+axis[2]*v[2]+axis[3]*v[3])*(1.0-cosphi);

    new[1] = axis[1]*dot+v[1]*cosphi+(-axis[3]*v[2]+axis[2]*v[3])*sinphi;
    new[2] = axis[2]*dot+v[2]*cosphi+(+axis[3]*v[1]-axis[1]*v[3])*sinphi;
    new[3] = axis[3]*dot+v[3]*cosphi+(-axis[2]*v[1]+axis[1]*v[2])*sinphi;
}




/* returns a unit vector in a random direction in arguments dx,dy,dz */
/* also returns a random magnitude within the unit sphere as a return value */
float uniform3Ddev(float *dx, float *dy, float *dz, long *seed)
{
    float ran1(long *idum);
    float dr;
    
    /* pick a random direction by cutting a sphere out of a cube */
    dr = 0;
    while(dr>1 || dr < 1e-2)
    {
        *dx = 2.1*(ran1(seed)-0.5);
        *dy = 2.1*(ran1(seed)-0.5);
        *dz = 2.1*(ran1(seed)-0.5);
        dr = sqrt(*dx**dx+*dy**dy+*dz**dz);
    }
    /* turn this into a unit vector */
    *dx/=dr;
    *dy/=dr;
    *dz/=dr;
    
    /* dx,dy,dz should now be a random unit vector */
    
    return dr;
}


float poidev(float xm, long *idum)
{
    float gammln(float xx);
    float ran1(long *idum);
    /* oldm is a flag for whether xm has changed since last call */
    static float sq,alxm,g,oldm=(-1.0);
    float em,t,y;
        
    if (xm < 12.0) {
        /* use direct method: simulate exponential delays between events */
        if(xm != oldm) {
            /* xm is new, compute the exponential */
            oldm=xm;
            g=exp(-xm);
        }
        /* adding exponential deviates is equivalent to multiplying uniform deviates */
        /* final comparison is to the pre-computed exponential */
        em = -1;
        t = 1.0;
        do {
            ++em;
            t *= ran1(idum);
        } while (t > g);
    } else {
        /* Use rejection method */
        if(xm != oldm) {
            /* xm has changed, pre-compute a few things... */
            oldm=xm;
            sq=sqrt(2.0*xm);
            alxm=log(xm);
            g=xm*alxm-gammln(xm+1.0);
        }
        do {
            do {
                /* y is a deviate from a lorentzian comparison function */
                y=tan(PI*ran1(idum));
                /* shift and scale */
                em=sq*y+xm;
            } while (em < 0.0);		/* there are no negative Poisson deviates */
            /* round off to nearest integer */
            em=floor(em);
            /* ratio of Poisson distribution to comparison function */
            /* scale it back by 0.9 to make sure t is never > 1.0 */
            t=0.9*(1.0+y*y)*exp(em*alxm-gammln(em+1.0)-g);
        } while (ran1(idum) > t);
    }
        
    return em;
}


/* return gaussian deviate with rms=1 and FWHM = 2/sqrt(log(2)) */
float gaussdev(long *idum)
{
    float ran1(long *idum);
    static int iset=0;
    static float gset;
    float fac,rsq,v1,v2;
        
    if (iset == 0) {
        /* no extra deviats handy ... */
        
        /* so pick two uniform deviates on [-1:1] */
        do {
            v1=2.0*ran1(idum)-1.0;
            v2=2.0*ran1(idum)-1.0;
            rsq=v1*v1+v2*v2;
        } while (rsq >= 1.0 || rsq == 0);
        /* restrained to the unit circle */
        
        /* apply Box-Muller transformation to convert to a normal deviate */
        fac=sqrt(-2.0*log(rsq)/rsq);
        gset=v1*fac;
        iset=1;		/* we now have a spare deviate */
        return v2*fac;
    } else {
        /* there is an extra deviate in gset */
        iset=0;
        return gset;
    }
}


/* generate Lorentzian deviate with FWHM = 2 */
float lorentzdev(long *seed) {
    float ran1(long *idum);
 
    return tan(M_PI*(ran1(seed)-0.5));
}

/* return triangular deviate with FWHM = 1 */
float triangledev(long *seed) {
    float ran1(long *idum);
    float value;

    value = ran1(seed);
    if(value > 0.5){
        value = sqrt(2*(value-0.5))-1;
    }else{
        value = 1-sqrt(2*value);
    }

    return value;
}



float expdev(long *idum)
{
    float dum;
    
    do
    dum=ran1(idum);
    while( dum == 0.0);
    return -log(dum);
}



/* ln of the gamma function */
float gammln(float xx)
{
    double x,y,tmp,ser;
    static double cof[6]={76.18009172947146,-86.50532032941677,
    24.01409824083091,-1.231739572450155,
    0.1208650973866179e-2,-0.5395239384953e-5};
    int j;
    
    y=x=xx;
    tmp=x+5.5;
    tmp -= (x+0.5)*log(tmp);
    ser = 1.000000000190015;
    for(j=0;j<=5;++j) ser += cof[j]/++y;
    
    return -tmp+log(2.5066282746310005*ser/x);
}





/* returns a uniform random deviate between 0 and 1 */
#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define EPS 1.2e-7
#define RNMX (1.0-EPS)

float ran1(long *idum)
{
    int j;
    long k;
    static long iy=0;
    static long iv[NTAB];
    float temp;
    
    if (*idum <= 0 || !iy) {
        /* first time around.  don't want idum=0 */
        if(-(*idum) < 1) *idum=1;
        else *idum = -(*idum);
        
        /* load the shuffle table */
        for(j=NTAB+7;j>=0;j--) {
            k=(*idum)/IQ;
            *idum=IA*(*idum-k*IQ)-IR*k;
            if(*idum < 0) *idum += IM;
            if(j < NTAB) iv[j] = *idum;
        }
        iy=iv[0];
    }
    /* always start here after initializing */
    k=(*idum)/IQ;
    *idum=IA*(*idum-k*IQ)-IR*k;
    if (*idum < 0) *idum += IM;
    j=iy/NDIV;
    iy=iv[j];
    iv[j] = *idum;
    if((temp=AM*iy) > RNMX) return RNMX;
    else return temp;
}


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

void polint(double *xa, double *ya, double x, double *y)
{
	double x0,x1,x2,x3;
	x0 = (x-xa[1])*(x-xa[2])*(x-xa[3])*ya[0]/((xa[0]-xa[1])*(xa[0]-xa[2])*(xa[0]-xa[3])); 
        x1 = (x-xa[0])*(x-xa[2])*(x-xa[3])*ya[1]/((xa[1]-xa[0])*(xa[1]-xa[2])*(xa[1]-xa[3]));
	x2 = (x-xa[0])*(x-xa[1])*(x-xa[3])*ya[2]/((xa[2]-xa[0])*(xa[2]-xa[1])*(xa[2]-xa[3]));
	x3 = (x-xa[0])*(x-xa[1])*(x-xa[2])*ya[3]/((xa[3]-xa[0])*(xa[3]-xa[1])*(xa[3]-xa[2]));
	*y = x0+x1+x2+x3;
}



void polin2(double *x1a, double *x2a, double **ya, double x1, double x2, double *y)
{
	void polint(double *xa, double *ya, double x, double *y);
	int j;
	double ymtmp[4];
	for (j=1;j<=4;j++) {
		polint(x2a,ya[j-1],x2,&ymtmp[j-1]);
	}
	polint(x1a,ymtmp,x1,y);
}


void polin3(double *x1a, double *x2a, double *x3a, double ***ya, double x1,
	double x2, double x3, double *y)
{
	void polint(double *xa, double ya[], double x, double *y);
	void polin2(double *x1a, double *x2a, double **ya, double x1,double x2, double *y);
	void polin1(double *x1a, double *ya, double x1, double *y);
	int j;
	double ymtmp[4];

	for (j=1;j<=4;j++) {
	    polin2(x2a,x3a,&ya[j-1][0],x2,x3,&ymtmp[j-1]);
	}
	polint(x1a,ymtmp,x1,y);
}


/* FWHM = integral = 1 */
double ngauss2D(double x,double y)
{
    return log(16)/M_PI*exp(-log(16)*(x*x+y*y));
}
double ngauss2Dinteg(double x,double y)
{
    return 0.125*(erf(2*x*sqrt(log(2)))*erf(y*sqrt(log(16)))*sqrt(log(16)/log(2)));
}

